(ns dda.build.c4k
  (:require
   [clojure.spec.alpha :as s]
   [clojure.spec.test.alpha :as st]
   [orchestra.core :refer [defn-spec]]
   [dda.build.devops :as devops]
   [dda.build.terragrunt :as tg]
   [dda.build.config :as cfg]
   [dda.build.c4k.domain :as domain]
   [dda.build.infrastructure :as i]))

(def default
  (merge devops/default
         {:c4k-output-prefix "c4k-app"
          :c4k-config-parts []
          :c4k-config-input "c4k-config.edn"
          :c4k-auth-input "c4k-auth.edn"}))

(s/def ::c4k (s/merge
              ::devops/devops
              (s/keys :opt-un [::domain/c4k-output-prefix 
                               ::domain/c4k-config-input 
                               ::domain/c4k-auth-input 
                               ::domain/c4k-app-name
                               ::domain/c4k-config-parts])))

(defn- default-cfg
  [input]
  (merge default
         (when (and (not (contains? input :c4k-app-name))
                    (contains? input :module))
           {:c4k-app-name (:module input)})
         input))

(defn-spec clean-build-dir! nil?
  [devops ::c4k]
  (let [config (default-cfg devops)]
    (i/execute! (domain/clean-build-dir-command config) config)))

(defn-spec run-c4k-jar! nil?
  [devops ::c4k]
  (let [config (default-cfg devops)]
    (doseq [c (domain/c4k-uberjar-command config)]
      (i/execute! c config))))

(defn-spec run-c4k-bin! nil?
  [devops ::c4k]
  (let [config (default-cfg devops)]
    (doseq [c (domain/c4k-graalvm-command config)]
      (i/execute! c config))))

(defn-spec write-c4k-objects! nil?
  [devops ::c4k
   tf-out ::tg/tf-out]
  (let [config (default-cfg devops)]
    (->> (domain/create-c4k-config 
          (cfg/read-config (domain/config-path config))
          config
          tf-out)
         (pr-str)
         (spit (domain/config-path config)))
    (->> (cfg/read-config (domain/auth-path config))
         (pr-str)
         (spit (domain/auth-path config)))
    (run-c4k-jar! config)))

(st/instrument `clean-build-dir!)
(st/instrument `run-c4k-jar!)
(st/instrument `run-c4k-bin!)
(st/instrument `write-c4k-objects!)
