(ns dda.build.c4k.domain
  (:require [clojure.spec.alpha :as s]
            [orchestra.core :refer [defn-spec]]
            [clojure.string :as cs]
            [dda.build.devops.domain :as d]
            [dda.build.terragrunt.domain :as td]))

(s/def ::c4k-app-name string?)
(s/def ::c4k-output-prefix string?)
(s/def ::c4k-config-input string?)
(s/def ::c4k-auth-input string?)
(s/def ::grafana-cloud-url string?)
(s/def ::c4k-config-part string?)
(s/def ::c4k-config-parts (s/* ::c4k-config-part))

(s/def ::config
  (s/merge ::d/devops
           (s/keys :req-un [::c4k-output-prefix ::c4k-config-input ::c4k-auth-input ::c4k-app-name ::c4k-config-parts]
                   :opt-un [::grafana-cloud-url ::grafana-cloud-user ::grafana-cloud-password])))

(defn-spec config-path string?
  [config ::config]
  (let [{:keys [c4k-config-input]} config]
    (str (d/build-path config) "/" c4k-config-input)))

(defn-spec auth-path string?
  [config ::config]
  (let [{:keys [c4k-auth-input]} config]
    (str (d/build-path config) "/" c4k-auth-input)))

(defn-spec output-path string?
  [config ::config]
  (let [{:keys [c4k-output-prefix]} config]
    (str (d/build-path config) "/" c4k-output-prefix ".yaml")))

(defn-spec partial-output-path string?
  [config ::config
   part ::c4k-config-part]
  (let [{:keys [c4k-output-prefix]} config]
    (str (d/build-path config) "/" c4k-output-prefix "-" part ".yaml")))

(defn-spec all-output-paths string?
  [config ::config]
  (let [{:keys [c4k-output-prefix, c4k-config-parts]} config]
    (if (empty? c4k-config-parts)
      (str (d/build-path config) "/" c4k-output-prefix ".yaml")
      (cs/join "," (map #(partial-output-path config %) c4k-config-parts)))
    ))

(defn-spec clean-build-dir-command seq?
  [config ::config]
  ["rm"  "-rf" (d/build-path (dissoc config :module))])

(defn-spec c4k-uberjar-command seq?
  [config ::config]
  (let [{:keys [c4k-app-name c4k-config-parts]} config
        executable-name (str "c4k-" c4k-app-name "-standalone.jar")]
    (if (empty? c4k-config-parts)
      [["bash" "-c" (str executable-name " " (config-path config) " " 
                         (auth-path config) " > " (output-path config))]]
      (map (fn [part]
             ["bash" "-c" (str executable-name " -cp " part " " (config-path config) " "
                               (auth-path config) " > " (partial-output-path config part))])
           c4k-config-parts)
      )))

(defn-spec c4k-graalvm-command seq?
  [config ::config]
  (let [{:keys [c4k-app-name c4k-config-parts]} config
        executable-name (str "c4k-" c4k-app-name)]
    (if (empty? c4k-config-parts)
      [["bash" "-c" (str executable-name " " (config-path config) " " (auth-path config) " > " (output-path config))]]
      (map (fn [part]
             ["bash" "-c" (str executable-name " -cp " part " " (config-path config) " "
                               (auth-path config) " > " (partial-output-path config part))])
           c4k-config-parts))))

(defn-spec create-c4k-config map?
  [module-config map?
   config ::config
   tf-out ::td/tf-out]
  (let [{:keys [name module stage]} config]
    (merge
     {:fqdn (:fqdn (:value (:out tf-out)))}
     (when (contains? config :grafana-cloud-url)
           {:mon-cfg {:grafana-cloud-url (:grafana-cloud-url config)
                      :cluster-name (str name "-" module)
                      :cluster-stage stage}})
     module-config)))
