(ns dda.build.provs.domain
  (:require [clojure.spec.alpha :as s]
            [orchestra.core :refer [defn-spec]]
            [dda.build.devops.domain :as d]
            [dda.c4k-common.predicate :as pred]
            [dda.build.c4k.domain :as c4k-d]
            [dda.build.terragrunt.domain :as td]))

; TODO: Use a better spec for emails
;       should be added to c4k-common, it seems common enough
(s/def ::email pred/bash-env-string?)
(s/def ::fqdn pred/fqdn-string?)
(s/def ::ipv4 pred/ipv4-string?)
(s/def ::ipv6 pred/ipv6-string?)
(s/def ::echo boolean?)
(s/def ::k3s-auth-input string?)
(s/def ::k3s-output-filename string?)
(s/def ::k3s-provision-user pred/bash-env-string?)
(s/def ::k3s-hcloudApiToken string?)
(s/def ::k3s-encryptionPassphrase string?)
(s/def ::k3s-grafana-url string?)
(s/def ::k3s-grafana-user string?)
(s/def ::k3s-grafana-token string?)
(s/def ::config
  (s/merge ::c4k-d/config
           (s/keys :req-un [::email ::echo 
                            ::k3s-output-filename ::k3s-auth-input ::k3s-provision-user]
                   :opt-un [::k3s-hcloudApiToken ::k3s-encryptionPassphrase 
                            ::k3s-grafana-user ::k3s-grafana-token ::k3s-grafana-url])))

(s/def ::node
  (s/keys :req-un [::ipv4 ::ipv6]))
(s/def ::letsencryptEndpoint pred/letsencrypt-issuer?)
(s/def ::certmanager
  (s/keys :req-un [::email ::letsencryptEndpoint]))
(s/def ::parameter string?)
(s/def ::source string?)
(defn k3s-credential? [input] (s/valid? (s/keys :req-un [::source ::parameter]) input))
(s/def ::hcloudApiToken k3s-credential?)
(s/def ::encryptionPassphrase k3s-credential?)
(s/def ::hetzner (s/keys :req-un [::hcloudApiToken ::encryptionPassphrase]))
(s/def ::user string?)
(s/def ::url string?)
(s/def ::password k3s-credential?)
(s/def ::cluster string?)
(s/def ::grafana (s/keys :req-un [::cluster]))
(s/def ::server-config
  (s/keys :req-un [::fqdn ::node ::certmanager ::echo]
          :opt-un [::hetzner ::grafana]))

(defn-spec auth-path string?
  [config ::config]
  (let [{:keys [k3s-auth-input]} config]
    (str (d/build-path config) "/" k3s-auth-input)))

(defn-spec output-path string?
  [config ::config]
  (let [{:keys [k3s-output-filename]} config]
    (str (d/build-path config) "/" k3s-output-filename)))

(defn-spec provs-server-command seq?
  [config ::config
   tf-out ::td/tf-out]
  (let [{:keys [k3s-provision-user]} config
        ipv4 (get-in tf-out [:out :value :ipv4])]
    [["provs-server.jar" "k3s" (str k3s-provision-user "@" ipv4) "-c" (output-path config) "-a" (c4k-d/all-output-paths config)]]))

(defn-spec create-k3s-config ::server-config
  [config ::config
   tf-out ::td/tf-out]
  (let [{:keys [stage email echo name module]} config
        letsencrypt-endpoint (if (= stage "prod") "prod" "staging")
        values (:value (:out tf-out))
        {:keys [fqdn ipv4 ipv6]} values]
    (merge
     {:fqdn fqdn
      :node {:ipv4 ipv4
             :ipv6 ipv6}
      :certmanager {:email email
                    :letsencryptEndpoint letsencrypt-endpoint}
      :echo echo}
     (when (and (contains? config :k3s-encryptionPassphrase)
                (contains? config :k3s-hcloudApiToken))
       {:hetzner
        {:hcloudApiToken {:source "PLAIN"
                          :parameter (:k3s-hcloudApiToken config)}
         :encryptionPassphrase {:source "PLAIN"
                                :parameter (:k3s-encryptionPassphrase config)}}})
     (when (and (contains? config :k3s-grafana-token)
                (contains? config :k3s-grafana-user)
                (contains? config :k3s-grafana-url))
       {:grafana {:cluster (str name "-" module)}}))))
