; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.crate.liferay.app
   (:require
     [clojure.string :as string]
     [schema.core :as s]
     [schema-tools.core :as st]
     [pallet.actions :as actions]
     [org.domaindrivenarchitecture.pallet.crate.liferay.schema :as schema]
     [org.domaindrivenarchitecture.pallet.crate.liferay.app-config :as app-config]
     [org.domaindrivenarchitecture.pallet.crate.liferay.db-replace-scripts :as db-replace-scripts]))

(defn- liferay-config-file
  "Create and upload a config file"
  [file-name content  & {:keys [owner mode]
            :or {owner "tomcat7" mode "644"}}]
  (actions/remote-file
    file-name
    :owner owner
    :group owner
    :mode mode
    :literal true
    :content (string/join \newline content))
  )

(defn- liferay-remote-file
  "Create and upload a config file"
  [file-name url & {:keys [owner mode]
                    :or {owner "tomcat7" mode "644"}}]
  (actions/remote-file
    file-name
    :owner owner
    :group owner
    :mode mode
    :insecure true
    :literal true
    :url url)
  )

(defn- liferay-dir
  "Create and upload a config file"
  [dir-path & {:keys [owner mode]
            :or {owner "tomcat7"
                 mode "755"}}]
  (actions/directory 
      dir-path
      :action :create
      :recursive true
      :owner owner
      :group owner
      :mode mode)
  )

(defn create-liferay-directories
  [liferay-home-dir liferay-lib-dir liferay-release-dir liferay-deploy-dir]
  (liferay-dir (str liferay-home-dir "logs"))
  (liferay-dir (str liferay-home-dir "data"))
  (liferay-dir liferay-deploy-dir)
  (liferay-dir liferay-lib-dir)
  (liferay-dir liferay-release-dir :owner "root")
  )

; TODO: review mje 18.08: Das ist tomcat spezifisch und gehört hier raus.
(defn delete-tomcat-default-ROOT
  [tomcat-root-dir]
  (actions/directory
    tomcat-root-dir
    :action :delete
    :recursive true)
  )

(defn liferay-portal-into-tomcat
  "make liferay tomcat's ROOT webapp"
  [tomcat-root-dir liferay-download-source]
  (actions/remote-directory 
    tomcat-root-dir
    :url liferay-download-source
    :unpack :unzip
    :recursive true
    :owner "tomcat7"
    :group "tomcat7")
  )

(defn liferay-dependencies-into-tomcat
  [liferay-lib-dir repo-download-source]
  "get dependency files" 
  (doseq [jar ["activation" "ccpp" "hsql" "jms" 
               "jta" "jtds" "junit" "jutf7" "mail" 
               "mysql" "persistence" "portal-service" 
               "portlet" "postgresql" "support-tomcat"]]
    (let [download-location (str repo-download-source jar ".jar")
          target-file (str liferay-lib-dir jar ".jar")]
      (liferay-remote-file target-file download-location)))
  )

(s/defn ^:allwas-validate release-dir :- s/Str
  ""
  [base-release-dir :- s/Str 
   release :- schema/LiferayRelease ]
  (str base-release-dir (st/get-in release [:name]) "-" (string/join "." (st/get-in release [:version]))))

(s/defn ^:always-validate prepare-apps-rollout :- s/Any
  "prepare the rollout of liferay applications"
  [dir :- s/Str 
   apps :- [schema/LiferayApp]]
  (liferay-dir dir :owner "root")
  (doseq [app apps]
    (liferay-remote-file (str dir (first app) ".war") (second app) :owner "root")
    )
  )

(s/defn ^:always-validate install-script-do-rollout
  "Creates script for rolling liferay version. To be called by the admin connected to the server via ssh"
  [liferay-home :- schema/NonRootDirectory
   prepare-dir :- schema/NonRootDirectory 
   deploy-dir :- schema/NonRootDirectory
   tomcat-dir :- schema/NonRootDirectory]
  (actions/remote-file
    (str liferay-home "do-rollout.sh")
    :owner "root"
    :group "root"
    :mode "0744"
    :literal true
    :content (app-config/rollout-script prepare-dir deploy-dir tomcat-dir)
    ))

(s/defn ^:always-validate prepare-rollout 
  "prepare the rollout of all releases"
  [liferay-release-config :- schema/LiferayReleaseConfig]
  (let [base-release-dir (st/get-in liferay-release-config [:release-dir])
        releases (st/get-in liferay-release-config [:releases])]
    (doseq [release releases]
      (let [release-dir (release-dir base-release-dir release)]
        (liferay-dir release-dir :owner "root")
        (prepare-apps-rollout (str release-dir "/app/") [(st/get-in release [:application])])
        (prepare-apps-rollout (str release-dir "/hooks/") (st/get-in release [:hooks]))
        (prepare-apps-rollout (str release-dir "/layouts/") (st/get-in release [:layouts]))
        (prepare-apps-rollout (str release-dir "/themes/") (st/get-in release [:themes]))
        (prepare-apps-rollout (str release-dir "/portlets/") (st/get-in release [:portlets]))
      ))
    ))

(s/defn install-liferay
  [tomcat-root-dir tomcat-webapps-dir liferay-home-dir liferay-lib-dir liferay-deploy-dir repo-download-source 
   liferay-release-config :- schema/LiferayReleaseConfig]
  "creates liferay directories, copies liferay webapp into tomcat and loads dependencies into tomcat"
  (create-liferay-directories liferay-home-dir liferay-lib-dir (st/get-in liferay-release-config [:release-dir]) liferay-deploy-dir)
  (delete-tomcat-default-ROOT tomcat-root-dir)
  (liferay-dependencies-into-tomcat liferay-lib-dir repo-download-source)
  (liferay-dependencies-into-tomcat liferay-lib-dir repo-download-source)
  (install-script-do-rollout liferay-home-dir (st/get-in liferay-release-config [:release-dir]) liferay-deploy-dir tomcat-webapps-dir)
  )

(defn configure-liferay
  [custom-build? & {:keys [db-name db-user-name db-user-passwd
                    portal-ext-properties fqdn-to-be-replaced fqdn-replacement]}]
  (let [effective-portal-ext-properties 
        (if (empty? portal-ext-properties) 
          (app-config/var-lib-tomcat7-webapps-ROOT-WEB-INF-classes-portal-ext-properties 
            :db-name db-name
            :db-user-name db-user-name
            :db-user-passwd db-user-passwd)
           portal-ext-properties)]
    
    (liferay-config-file
      (if custom-build?
        "/var/lib/liferay/portal-ext.properties"
        "/var/lib/tomcat7/webapps/ROOT/WEB-INF/classes/portal-ext.properties")
       effective-portal-ext-properties)
    
    (liferay-config-file 
      "/var/lib/liferay/prodDataReplacements.sh"
      (db-replace-scripts/var-lib-liferay-prodDataReplacements-sh
        fqdn-to-be-replaced fqdn-replacement db-name db-user-name db-user-passwd)
      :owner "root" :mode "744"))
  )