; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.crate.d2rq
  (:require
    [schema.core :as s]
    [pallet.crate :as crate]
    [org.domaindrivenarchitecture.pallet.core.dda-crate :as dda-crate]
    [org.domaindrivenarchitecture.pallet.crate.tomcat :as tomcat]
    [org.domaindrivenarchitecture.pallet.crate.d2rq.app :as d2rq-app]
    ))

(def facility :dda-d2rq)

(defn- d2rq-base [] 
  "/var/lib/d2rq")

(def D2rqSchema
  {:tomcat tomcat/TomcatConfig
   :linked-data-war-url s/Str
   })

(def default-config
  {:tomcat
   (tomcat/merge-config 
     {:server-xml-config
      {:shutdown-port "8006"
       :connector-protocol "AJP/1.3"
       :connector-port "8019"}
      :java-vm-config 
      {:xmx "1024m"
       :xms "256m"
       :max-perm-size "128m"}
      :custom-config 
      {:custom-tomcat-home (str (d2rq-base) "/tomcat7/")}})
   })

(defn install
  [app-name config
   & {:keys [webapp-name]
      :or {webapp-name "d2rq"}}]
  (tomcat/install (get-in config [:tomcat]))
  (d2rq-app/install-d2rq-war-file 
    (get-in config [:tomcat :custom-config :custom-tomcat-home])
    (get-in config [:linked-data-war-url])
    :webapp-name webapp-name))

(defn configure
  [app-name config]                      
  (tomcat/configure (get-in config [:tomcat])))

(defmethod dda-crate/dda-install 
  facility [dda-crate partial-effective-config]
  (let [config (dda-crate/merge-config dda-crate partial-effective-config)]
    (install (name facility) config)))

(defmethod dda-crate/dda-configure 
  facility [dda-crate partial-effective-config]
  (let [config (dda-crate/merge-config dda-crate partial-effective-config)]
    (configure (name facility) config)))

(def d2rq-crate
  (dda-crate/make-dda-crate
    :facility facility
    :version [0 1 0]
    :config-schema D2rqSchema
    :config-default default-config))

(def with-d2rq
  (dda-crate/create-server-spec d2rq-crate))
