; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.crate.tomcat 
  (:require
    [schema.core :as s]
    [pallet.api :as api]
    [org.domaindrivenarchitecture.pallet.core.dda-crate :as dda-crate]
    [org.domaindrivenarchitecture.config.commons.map-utils :as map-utils]
    [org.domaindrivenarchitecture.pallet.crate.config :as config]
    [org.domaindrivenarchitecture.config.commons.directory-model :as dir-model]
    [org.domaindrivenarchitecture.pallet.crate.tomcat.app :as app]
    [org.domaindrivenarchitecture.pallet.crate.tomcat.app-config :as app-config]
    ))

; TODO: review jem 2016.05.28: home-dir is someway redundant to custom-config. But lets keep this refactoring for future.
; TODO: review jem 2016.05.28: TomcatConfig is no more compatible. So you have to refactor every point of usage according to new config structure. 
(def TomcatConfig
  "The configuration for tomcat crate." 
  {:home-dir dir-model/NonRootDirectory
   :webapps-dir dir-model/NonRootDirectory
   :server-xml-config app-config/ServerXmlConfig
   :java-vm-config app-config/JavaVmConfig
   :custom-config app-config/CustomConfig
   })

(def default-config
  "Tomcat Crate Default Configuration"
  {:home-dir "/var/lib/tomcat7/"
   :webapps-dir "/var/lib/tomcat7/webapps/"
   :server-xml-config app-config/default-server-xml-config
   :java-vm-config app-config/default-heap-config
   :custom-config app-config/default-custom-config})

(s/defn ^:always-validate merge-config :- TomcatConfig
  "merges the partial config with default config & ensures that resulting config is valid."
  [partial-config]
  (map-utils/deep-merge default-config partial-config))

(s/defn install
  "install function for httpd-crate."
  [config :- TomcatConfig]
  (app/install-tomcat7 config))

(s/defn configure
  "configure function for httpd-crate."
  [config :- TomcatConfig]
  (app/configure-tomcat7 config))
  
(defmethod dda-crate/dda-install 
  :dda-tomcat [dda-crate partial-effective-config]
  (let [config (dda-crate/merge-config dda-crate partial-effective-config)]
    (install config)))

(defmethod dda-crate/dda-configure 
  :dda-tomcat [dda-crate partial-effective-config]
  (let [config (dda-crate/merge-config dda-crate partial-effective-config)]
    (configure config)))

(def dda-tomcat-crate 
  (dda-crate/make-dda-crate
    :facility :dda-tomcat
    :version [0 1 0]
    :config-schema TomcatConfig
    :config-default default-config
    ))

(def with-tomcat
  (dda-crate/create-server-spec dda-tomcat-crate))