(ns clean-latex.core
  (:require [clean-latex.helpers :refer [fidentity map-lines]]
            [clojure.edn :as edn]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [me.raynes.fs :as fs])
  (:gen-class))

;;; remove latex comments

(defn remove-comment [s]
  (let [comment-re #"%.*"
        comment-blank "%"]
    (str/replace s comment-re comment-blank)))

(def remove-comments
  "remove all comments from latex code while preserving whitespace in final document"
  (partial map-lines remove-comment))

;;; empty out particular macros

(defn empty-macro
  "empty out every occurance of macro-name"
  [s macro-name]
  (let [macro-re (re-pattern (str "\\\\" macro-name "\\{[^{]*?\\}"))]
    (str/replace s macro-re (str "\\\\" macro-name "{}"))))

(defn nonempty-macro
  "find non-empty all occurances of macro-name"
  [s macro-name]
  (let [macro-re (re-pattern (str "\\\\" macro-name "\\{[^}]+?\\}"))]
    (re-seq macro-re s)))

;;; empty out particular environments (e.g, comment envirnoment)

;; regex modifier that causes dot to match also newlines
(def dotall "(?s)")

(defn empty-environment
  "empty out every occurance of env-name"
  [s env-name]
  (let [env-begin (format "\\\\begin\\{%s\\}" env-name env-name)
        env-end (format "\\\\end\\{%s\\}" env-name env-name)
        env-re (re-pattern (str dotall env-begin "(.*?)" env-end))
        env-empty (format "\\begin{%s}\n\\end{%s}" env-name env-name)]
    (str/replace s env-re (fn [[matched inner]]
                            (if (re-find (re-pattern env-begin) inner)
                              (do
                                (println "[report] " matched)
                                matched)
                              env-empty)))))

;;; clean up latex source

(defn clean-latex
  "empty out occurances of macros in macro-names and remove comments"
  [{:keys [env-names macro-names] :as config} s]
  (as-> s s
        (reduce empty-macro s macro-names)
        (reduce empty-environment s env-names)
        (remove-comments s)))

;;; copy files to clean-dir

(defn copy-file [{:keys [root clean-dir] :as config} f]
  (println "copy: " f)
  (fs/mkdirs (fs/parent (io/file clean-dir f)))
  (fs/copy (io/file root f) (io/file clean-dir f)))

(defn copy-files
  "copy files-to-copy from root to clean-dir"
  [{:keys [files-to-copy] :as config}]
  (->> files-to-copy
       (map (partial copy-file config))
       (dorun)))

;;; report uncleaned latex

(defn report-latex
  "report non-empty occurances of macros in macro-names or envs in env-names"
  [{:keys [macro-names env-names] :as config} s]
  (->> (mapcat (partial nonempty-macro s) macro-names)
       (map (partial println "report: "))
       (dorun)))

;;; clean latex source files and write to clean-dir

(defn clean-file [{:keys [root clean-dir] :as config} f]
  (let [target-clean-file (io/file clean-dir f)]
    (fs/mkdirs (fs/parent target-clean-file))
    (->> (io/file root f)
       (fidentity (partial println "clean: "))
       (slurp)
       (clean-latex config)
       (fidentity (partial report-latex config))
       (spit target-clean-file))))

(defn clean-files
  "clean and write files-to-clean; report uncleaned latex code"
  [{:keys [files-to-clean] :as config}]
  (->> files-to-clean
       (map (partial clean-file config))
       (dorun)))

;;; read config, clean files, and copy files

(defn -main
  "clean and copy files as specified in config file"
  [& args]
  (let [config-file (or (first args) "config.edn")]
    (->> config-file
         (slurp)
         (edn/read-string)
         (fidentity (partial clean-files))
         (fidentity (partial copy-files))
         )
    ))
