(ns component.layout
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [com.stuartsierra.component :as component]
            [hiccup.page :refer [html5 include-css include-js]]
            [meta-merge.core :as m]))

;; TODO: spec for options
;; bootstrap as dependency?

(defn- meta-data
  [{:keys [title date author description] :as options}]
  (list
   [:meta {:charset "utf-8"}]
   [:meta {:name "viewport"
           :content "width=device-width, initial-scale=1.0, user-scalable=yes, shrink-to-fit=no"}]
   [:meta {:http-equiv "x-ua-compatible" :content "ie=edge"}]
   (when author
     [:meta {:name "author" :content author}])
   (when (seq description)
     [:meta {:name "description"
             :content description}])
   (when date
     ;; which date format?
     [:meta {:name "dcterms.date"
             :content date}])))

(defn- css [{:keys [css css-hrefs] :as options}]
  (list
   (apply include-css css-hrefs)
   (when css
     [:style css])))

(defn- head [{:keys [title title-short site-title] :as options}]
  [:head
   [:title (if title-short
             (str site-title " » " title-short)
             title)]
   (meta-data options)
   (css options)])

(defn- navbar [{:keys [home-uri site-title menu] :as options}]
  [:nav #_.navbar.navbar-light.bg-faded
   [:div.container
    [:div.row
     [:div.col-md-4
      [:div.nav #_.navbar-nav
       [:a.nav-link.active #_.nav-item. {:href home-uri}
        site-title]]]
     [:div.col-md-7.col-md-offset-1
      #_"test"
      [:div.nav.nav-inline #_.navbar-nav
       (for [{:keys [label uri active] :as item} menu]
         [:a {:href uri
              :class (str/join " " ["nav-item" "nav-link" (when active "active")])}
          label
          (when active
            [:span.sr-only "(current)"])])]]]]])

(defn- snippet [r]
  (slurp (io/resource r)))

(defn- render*
  [{:keys [js-paths body snippets script] :as options}]
  (html5
   (head options)
   [:body.rhythm
    (map snippet snippets)
    (navbar options)
    [:div.container
     [:div.row
      [:div.col-md-12
       body]]]
    (apply include-js js-paths)
    [:script script]]))

(defn render [component options]
  (render* (m/meta-merge component options)))

(defrecord Layout []
  component/Lifecycle
  (start [this]
    (println "[layout] start")
    this)
  (stop [this]
    (println "[layout] stop")
    this))

(defn layout [& [m]]
  (map->Layout m))
