(ns component.sass
  (:require [clojure.java.io :as io]
            [clojure.string :as str]
            [com.stuartsierra.component :as component]
            [component.modified :as modified]
            [sass4clj.core :as sass]
            [schema.core :as s])
  (:import java.io.File))

;; TODO: use spec instead of schema

(defn- main-file?
  "check if file is a main sass/scss file (not a partial)"
  [file]
	(and
   (or
    (.endsWith (.getName file) ".scss")
    (.endsWith (.getName file) ".sass") )
   (not (.startsWith (.getName file) "_"))))

(defn- find-main-files
  "find all main files in paths"
  [source-path]
  (let [file (io/file source-path)]
    (->> (file-seq file)
         (filter main-file?)
         (map (fn [^File x] [(.getPath x)
                            (.toString (.relativize (.toURI file) (.toURI x)))])))))

(defn compile-sass [{:keys [target-path source-paths] :as options}]
	(doseq [[path relative-path] (mapcat find-main-files source-paths)
          :let [output-rel-path (str/replace relative-path #"\.(sass|scss)$" ".css")
                output-path     (.getPath (io/file target-path output-rel-path))]]
		(println (format "Compiling Sass: %s" relative-path))
    (as-> options _
      (dissoc _ :target-path :source-paths)
      (sass4clj.core/sass-compile-to-file path output-path _))))



(defn out-of-date? [{:keys [last-sass-file source-paths] :as config}]
  (modified/out-of-date? (mapcat (comp file-seq io/file) source-paths)
                         last-sass-file))

(defn touch-file [{:keys [last-sass-file] :as config}]
  (io/make-parents (io/file last-sass-file))
  (spit (io/file last-sass-file)
        (prn-str (java.util.Date.))))

;;; component

(s/defrecord Sass [target-path :- s/Str
                   check-date? :- s/Bool
                   source-paths :- [s/Str]
                   last-sass-file :- s/Str
                   output-style :- (s/enum :nested :compact :expanded :compressed)
                   verbosity :- (s/enum 1 2)]
	component/Lifecycle
  (start [this]
    (println "[sass]" "start")
    (cond-> this
      (or (not (:check-date? this)) (out-of-date? this))
      (doto
        (touch-file)
        (compile-sass))))
  (stop [this]
    this))

(defn sass [& [options]]
  (-> {:verbosity 1
       :output-style :compressed}
      (merge options)
      (map->Sass)))



;;; tests

#_(sass {:target-path "target"})

#_(-> reloaded.repl/system :sass :source-paths first io/file file-seq)

#_(-> reloaded.repl/system :sass :last-sass-file)

#_(-> reloaded.repl/system :sass out-of-date?)
