(ns deploy.component.shell
  (:require [com.stuartsierra.component :as component]
            [clojure.string :as str]
            [clojure.java.shell :as sh]
            [clojure.spec :as s]
            [clojure.spec.test :as stest]))



(s/def ::print-cmd? boolean?)
(s/def ::print-out? boolean?)
(s/def ::print-err? boolean?)
(s/def ::abort-on-error? boolean?)

(s/def ::config
  (s/keys :req-un [::print-cmd?
                   ::print-out?
                   ::print-err?
                   ::abort-on-error?]))

(s/def ::std
  ::config)
(s/def ::no-abort
  ::config)
(s/def ::no-out
  ::config)


(s/def ::args
  (s/cat
   :config ::config
   :args (s/* string?)))

(s/fdef exec
        :args ::args
        :fn nil?)

#_(stest/instrument [`exec `map->Shell])



(defn default-config []
  {:print-cmd?      true
   :print-err?      true
   :print-out?      true
   :abort-on-error? true})



(defn exec
  [{:keys [print-cmd? print-out? print-err? abort-on-error?] :as config}
   & args]
  (let [{:keys [out exit err] :as res} (apply sh/sh args)]
    ;; log result of command
    (when print-cmd?
      (println "Command:" (str/join " " args) (format "(%d)" exit)))
    (when (seq out)
      (if print-out?
        (println out)
        (println "Output:" (count (str/split-lines out)) "lines")))
    (when (seq err)
      (binding [*out* *err*]
        (if print-err?
          (println err)
          (println "Error:" (count (str/split-lines err)) "lines"))))
    ;; abort on error
    (when-not (= exit 0)
      (when abort-on-error?
        (throw (ex-info "shell command failed"
                        {:args args
                         :res res}))))))



(defrecord Shell []
  component/Lifecycle
  (start [component]
    (println "[shell] start")
    component)
  (stop [component]
    (println "[shell] stop")
    component))

(defn shell [& [m]]
  (->> m
       (merge (default-config))
       (map->Shell)))
