(ns doctex.document
  (:require [clojure.pprint :as pp]
            [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [clojure.java.io :as io]
            [com.stuartsierra.dependency :as dep]
            [doctex
             [core :as core]
             [specs :as specs]
             [util :as util]]
            [duct
             [core :as duct]
             [logger :refer [log]]]
            [integrant.core :as ig]
            [selmer.parser :as selmer]
            [clojure.edn :as edn])
  (:import java.io.FileNotFoundException))

(defn dependencies [config]
  (for [[node metadata] config
        dep             (map keyword (:inherit metadata))]
    (do
      (when-not (contains? config dep)
       (throw
        (ex-info (format ":inherit key %s for %s not found"
                         (name dep) (name node))
                 {:missing-key dep
                  :key         node}))      )
     [node dep])))

(defn graph [edges]
  (reduce
   (fn [g [node dep]]
     (dep/depend g node dep))
   (dep/graph)
   edges))

(defn dependency-order [config]
  (-> config
      dependencies
      graph
      dep/topo-sort))

(defn merge-document-configs [config k]
  (->>
   (concat (map keyword (get-in config [k :inherit])) [k])
   (map #(get config %))
   (apply merge)))

(defn merge-all-document-configs [config]
  (reduce
   (fn [config k]
     (assoc config k (merge-document-configs config k)))
   config
   (dependency-order config)))

;;; specs

(s/def ::document-path
  ::specs/path-string)

(s/def ::template-path
  ::specs/path-string)

(s/def ::readconfig
  (s/and ::specs/join-paths
         ::specs/exists-path
         ;; use yaml instead
         ::specs/slurp-yaml
         (s/conformer
          merge-all-document-configs)
         (s/conformer
          (fn group-by-priority [m]
            (group-by (comp boolean :priority val) m)))
         (s/conformer
          (fn sort-by-priority [m]
            (concat (get m true) (get m false))))
         (s/conformer
          (fn remove-hidden [col]
            (remove (comp :hidden second) col)))
         (s/conformer
          (fn remove-implicit-hidden [col]
            (remove (comp #(str/starts-with? % "_") name first) col)))))

(s/def :doctex.document.conform/template
  string?)

#_(s/def ::files
  (s/and
   (s/or
    :nested (s/map-of keyword? (s/nilable (s/coll-of string?)))
    :flat (s/coll-of string?))
   (s/conformer second)))

;; TODO: manually check that files exist; latex doesn't do that nicely

(s/def ::document
  (s/cat
   :key keyword?
   :body (s/keys
          :req-un [:doctex.document.conform/template #_::files])))

(s/def ::template
  (s/and (specs/get-key :document)
         ::document
         (specs/get-key :body)
         (specs/get-key :template)))

(s/def ::document-context
  (s/and (specs/get-key :document)
         ::document
         (specs/get-key :body)))

(s/def ::key
  (s/and (specs/get-key :document)
         ::document
         (specs/get-key :key)))

(s/def ::doctex-context map?)

(util/derive-all
 {::template         ::specs/conform
  ::doctex-context   ::specs/assert
  ::document         ::specs/assert
  ::key              ::specs/conform
  ::document-context ::specs/conform})

;;; integrant

(defmethod ig/init-key
  ::readconfig
  [k paths]
  (fn []
    (try
      (s/conform ::readconfig paths)
      (catch Throwable t
        (throw
         (ex-info
          (format "failed to read config %s"
                  (str (apply io/file paths)))
          {:paths paths
           :key   k}
          t))))))

(defn remove-empty-lines [s]
  (as-> s _
    (str/split-lines _)
    (map str/trimr _)
    (remove str/blank? _)
    (str/join "\n" _)))

(defn remove-whitespace [s]
  (str/replace s #"(?m)(?s)\s*" ""))


(defmethod ig/init-key
  ::selmer
  [_ {:keys [template-path-url logger template context]}]

  (util/assert ::core/template-path-url template-path-url)
  (log logger :info ::selmer {:template-path-url (str template-path-url)
                              :template          template})

  (selmer/add-tag! :remove-empty-lines
                   (fn [args context-map content]
                     (remove-empty-lines (get-in content [:remove-empty-lines :content])))
                   :end-remove-empty-lines)

  (selmer/add-tag! :remove-whitespace
                   (fn [args context-map content]
                     (remove-whitespace (get-in content [:remove-whitespace :content])))
                   :end-remove-whitespace)

  (selmer/render-file
   ;; HACK: trick selmer into thinking that everyting is safe
   ;; not clear if that continues to work in future versions of selmer
   template (assoc context :selmer.filter-parser/selmer-safe-filter true)
   {:custom-resource-path template-path-url
    ;; :short-comment-second nil ;; what is this for? what is default
    :cache                false ;; default true
    :tag-open             \<
    :tag-close            \>
    :tag-second           \!
    :filter-open          \<
    :filter-close         \>}))

(s/def ::extension
  (s/conformer
   #(get {:latex   ".tex"
          :pdf     ".pdf"
          :synctex ".synctex.gz"}
         %
         ::s/invalid)))

(defmethod ig/init-key
  ::router
  [_ {:keys [document-path logger]}]

  (log logger :info ::router document-path)
  (fn router
    ([k] (str (io/file document-path)))
    ([k ext]
     (log logger :info ::router [k ext])
     (util/assert ::extension ext ::router)
     (str (io/file document-path (str (name k) (s/conform ::extension ext)))))))

(defn maybe-slurp [url]
  (try
    (slurp url)
    (catch FileNotFoundException _
        nil)))

(defmethod ig/init-key
  ::template-context
  [_ {:keys [logger template-path-url template]}]
  (log logger :info ::template-context [template-path-url template])
  (as->
      (str template-path-url "/" (str/replace template #"\.[a-z]*$" ".edn"))
      _
    (util/assert ::specs/url _ ::template-context)
    (do (log logger :info ::template-context-string _) _)
    (s/conform ::specs/url _)
    (do (log logger :info ::template-context-url _) _)
    ;; check if url resource exists
    (maybe-slurp _)
    (edn/read-string _)))

(defmethod ig/init-key
  ::spit
  [_ {:keys [router selmer key logger]}]
  (log logger :info ::spit [key (count selmer)])
  (util/spit-with-parents (router key :latex) selmer))

(defmethod ig/init-key
  ::copy-main
  [_ {:keys [context logger key router root]}]
  (when (:document/main context)
    (log logger :info ::copy key)
    (io/copy
     (io/file (router key :latex))
     (io/file root "main.tex"))))

(defmethod ig/init-key
  ::render
  [_ {:keys [config config-file router logger]}]

  (log logger :info ::render (keys config))

  (fn document-render [document]

    (log logger :info ::document-render
         (str "\n" (str/trim (with-out-str (pp/pprint document)))))

    (log logger :report :render
         (format "%s at %s" (first document) (router (first document) :latex)))

    (when-not (s/valid? ::document document)
      (throw
       (ex-info
        (format "\"%s\" invalid in %s: %s"
                (first document)
                config-file
                (some->>
                 document
                 (s/explain-data ::document)
                 (::s/problems)
                 (mapv :pred)))
        {:explain-data (s/explain-data ::document document)
         :key (first document)
         :document document})))

    (-> config
        (assoc ::document document)
        (duct/prep)
        (ig/init))))

(defn qualify-keys [m ns]
  (zipmap (map #(keyword ns (name %)) (keys m)) (vals m)))

(defmethod ig/init-key
  ::render-all
  [_ {:keys [render logger readconfig]}]

  (log logger :info ::render-all)

  (doto
      (fn render-all [& _]
        (doseq [document (readconfig)]
          (render document)))
    (.invoke)))

(util/derive-all
 {::render-config ::util/unwrap})

(defmethod ig/init-key
  ::context
  [_ m]
  (util/assert (s/spec (s/map-of keyword? (s/nilable map?))) m ::context)
  (->> m
       (map (fn [[k v]]
              (qualify-keys v (name k))))
       (apply merge)))
