(ns doctex.document
  (:require [clojure
             [edn :as edn]
             [pprint :as pp]
             [spec :as s]
             [string :as str]
             [walk :as walk]]
            [clojure.java.io :as io]
            [com.stuartsierra.dependency :as dep]
            [doctex
             [specs :as specs]
             [util :as util]]
            [duct.logger :refer [log]]
            [integrant.core :as ig]
            [selmer.parser :as selmer]
            [yaml.core :as yaml]
            [duct.core :as duct]))

(defn dependencies [config]
  (for [[node metadata] config
        dep (map keyword (:inherit metadata))]
    [node dep]))

(defn graph [edges]
  (reduce
   (fn [g [node dep]]
     (dep/depend g node dep))
   (dep/graph)
   edges))

(defn dependency-order [config]
  (-> config
      dependencies
      graph
      dep/topo-sort))

(defn merge-document-configs [config k]
  (->>
   (concat (map keyword (get-in config [k :inherit])) [k])
   (map #(get config %))
   (apply merge)))

(defn merge-all-document-configs [config]
  (reduce
   (fn [config k]
     (assoc config k (merge-document-configs config k)))
   config
   (dependency-order config)))

;;; specs

(s/def ::document-path
  ::specs/path-string)

(s/def ::template-path
  ::specs/path-string)

(s/def ::readconfig
  (s/and ::specs/join-paths
         ::specs/exists-path
         ;; use yaml instead
         ::specs/slurp-yaml
         (s/conformer
          merge-all-document-configs)
         (s/conformer
          (fn group-by-priority [m]
            (group-by (comp boolean :priority val) m)))
         (s/conformer
          (fn sort-by-priority [m]
            (concat (get m true) (get m false))))
         (s/conformer
          (fn remove-hidden [col]
            (remove (comp :hidden second) col)))))

(s/def :doctex.document.conform/template
  string?)

(s/def ::files
  (s/map-of keyword? (s/coll-of string?)))

(s/def ::document
  (s/cat
   :key keyword?
   :body (s/keys
          :req-un [:doctex.document.conform/template ::files])))

(s/def ::template
  (s/and (specs/get-key :document)
         ::document
         (specs/get-key :body)
         (specs/get-key :template)))

(s/def ::document-context
  (s/and (specs/get-key :document)
         ::document
         (specs/get-key :body)
         #_(specs/get-key :metadata)))

(s/def ::key
  (s/and (specs/get-key :document)
         ::document
         (specs/get-key :key)))

(s/def ::doctex-context map?)

(util/derive-all
 {::readconfig       ::specs/conform-fn
  ::template         ::specs/conform
  ::doctex-context   ::specs/assert
  ::document         ::specs/assert
  ::key              ::specs/conform
  ::document-context ::specs/conform})

;;; integrant

(defn remove-empty-lines [s]
  (as-> s _
    (str/split-lines _)
    (map str/trimr _)
    (remove str/blank? _)
    (str/join "\n" _)))

(defn remove-whitespace [s]
  (str/replace s #"(?m)(?s)\s*" ""))


(defmethod ig/init-key
  ::selmer
  [_ {:keys [template-path logger template context]}]

  (util/assert ::template-path template-path)
  (log logger :info ::selmer {:template-path template-path
                              :template template})

  (selmer/add-tag! :remove-empty-lines
                   (fn [args context-map content]
                     (remove-empty-lines (get-in content [:remove-empty-lines :content])))
                   :end-remove-empty-lines)

  (selmer/add-tag! :remove-whitespace
                   (fn [args context-map content]
                     (remove-whitespace (get-in content [:remove-whitespace :content])))
                   :end-remove-whitespace)

  (selmer/render-file
   ;; HACK: trick selmer into thinking that everyting is safe
   ;; not clear if that continues to work in future versions of selmer
   template (assoc context :selmer.filter-parser/selmer-safe-filter true)
   {:custom-resource-path (.toURL (io/file template-path))
    ;; :short-comment-second nil ;; what is this for? what is default
    :cache false ;; default true
    :tag-open \<
    :tag-close \>
    :tag-second \!
    :filter-open \<
    :filter-close \>}))

(s/def ::extension
  (s/conformer
   #(get {:latex   ".tex"
          :pdf     ".pdf"
          :synctex ".synctex.gz"}
         %
         ::s/invalid)))

(defmethod ig/init-key
  ::router
  [_ {:keys [document-path logger]}]

  (log logger :info ::router document-path)
  (fn router
    ([k] (str (io/file document-path)))
    ([k ext]
     (log logger :info ::router [k ext])
     (util/assert ::extension ext ::router)
     (str (io/file document-path (str (name k) (s/conform ::extension ext)))))))

(defmethod ig/init-key
  ::template-context
  [_ {:keys [logger template-path template]}]
  (log logger :info ::template-context [template-path template])
  (->>
   (io/file template-path (str/replace template #"\.[a-z]*$" ".edn"))
   str
   (s/conform ::specs/maybe-slurp-edn)))

(defmethod ig/init-key
  ::spit
  [_ {:keys [router selmer key logger]}]
  (log logger :info ::spit [key (count selmer)])
  (util/spit-with-parents (router key :latex) selmer))

(defmethod ig/init-key
  ::copy-main
  [_ {:keys [context logger key router root]}]
  (when (:document/main context)
    (log logger :info ::copy key)
    (io/copy
     (io/file (router key :latex))
     (io/file root "main.tex"))))

(defmethod ig/init-key
  ::render
  [_ {:keys [config router logger]}]

  (log logger :info ::render (keys config))

  (fn document-render [document]

    (log logger :info ::document-render (str "\n" (str/trim (with-out-str (pp/pprint document)))))

    (log logger :report :render (router (first document) :latex))

    (-> config
        (assoc ::document document)
        (duct/prep)
        (ig/init))))

(defn qualify-keys [m ns]
  (zipmap (map #(keyword ns (name %)) (keys m)) (vals m)))

(defmethod ig/init-key
  ::render-all
  [_ {:keys [render logger readconfig]}]

  (log logger :info ::render-all)

  (doto
      (fn render-all [& _]
        (doseq [document (readconfig)]
          (render document)))
    (.invoke)))

(util/derive-all
 {::render-config ::util/unwrap})

(defmethod ig/init-key
  ::context
  [_ m]
  (util/assert (s/spec (s/map-of keyword? (s/nilable map?))) m ::context)
  (->> m
       (map (fn [[k v]]
              (qualify-keys v (name k))))
       (apply merge)))
