(ns latex-compile.system
  (:gen-class)
  (:require [clojure.core.async :as a]
            [clojure.java.io :as io]
            clojure.stacktrace
            [com.stuartsierra.component :as component]
            [juxt.dirwatch :as dirwatch]
            [latex-compile
             [bibtex :as bibtex]
             [helpers :as helpers]
             [latex :as latex]]
            [me.raynes.fs :as fs]))

;;; helpers for xforms

(defn file-extension-map [f]
  (comp f fs/extension :file))

(defn file-name-map [f]
  (comp f fs/name :file))

(defn file-extension-cache-filter [f]
  (helpers/cache-filter (file-extension-map f)
                        (comp helpers/slurp-maybe :file)))

;;; protocols

(defprotocol IRun
  (run [this]))

(defprotocol IWatchRun
  (watch-ch [this]))

;;; components

(defrecord WatchRun [xfs]
  IWatchRun
  (watch-ch [this]
    (assert (:ch this))
    (:ch this))
  component/Lifecycle
  (start [this]
    (let [runners (->> (vals this)
                       (filter (partial satisfies? IRun)))
          run (fn [x] (try (run x)
                           (catch Throwable tr
                             (clojure.stacktrace/print-stack-trace tr))))]
      (assoc this
        :ch (a/chan (a/sliding-buffer 1)
                    (comp
                     (apply comp xfs)
                     (map (fn [_]
                            (->> runners
                                 (map run)
                                 (dorun))
                            true)))
                    ))))
  (stop [this]
        (some-> (:ch this)
                (a/close!))
        (dissoc this :ch)))

(defn new-watch-run [& xfs]
  (->WatchRun xfs))

(defn maybe-copy [from to]
  (when (fs/exists? from)
    (fs/copy from to)))

(defrecord Latex [opts]
  IRun
  (run [this]
    (let [{:keys [root master cmd temp-relative format]} opts]
      (latex/latex-single
       {:file (str root master ".tex")
        :cmd cmd
        :format format
        :temp-dir (str root temp-relative)})
      (let [synctex-ext "synctex.gz"]
        (maybe-copy (str root temp-relative master "/" master "." format)
                    (str root master "." format))
        (maybe-copy (str root temp-relative master "/" master "." synctex-ext)
                    (str root master "." synctex-ext))
        #_(when (fs/exists? from)
          (fs/copy from to))
        #_(when (fs/exists? from-synctex)
          (fs/copy from-synctex to-synctex))))))

(defn new-latex-component [opts]
  (->Latex opts))

(defrecord Bibtex [opts]
  IRun
  (run [this]
    (let [{:keys [root master temp-relative]} opts]
      (bibtex/bibtex
       {:file (str root temp-relative master "/" master ".aux")
        :source-file (str root master ".tex")}))))

(defn new-bibtex-component [opts]
  (->Bibtex opts))

(defrecord Watcher [opts xfs]
  component/Lifecycle
  (start [this]
    (let [xfs (apply comp
                     (remove (file-name-map (partial re-find (:exclude opts))))
                     (filter (file-extension-map (:extensions opts)))
                     xfs)
          ch (a/chan (a/sliding-buffer 1) xfs)
          mult (a/mult ch)]

      ;; tap watch-ch of IWatchRun instances to mult
      (->> (vals this)
           (filter (partial satisfies? IWatchRun))
           (map #(a/tap mult (watch-ch %)))
           (dorun))

      (assoc this
        :ch ch
        :ws (dirwatch/watch-dir #(a/>!! ch %)
                                (io/file (:root opts))))))
  (stop [this]
    (some-> (:ch this)
            (a/close!))
    (some-> (:ws this)
            (dirwatch/close-watcher))
    (dissoc this :ws :ch)))

(defn new-watcher-component [opts & xfs]
  (->Watcher opts xfs))

;;; system

(def default-config
  {:temp-relative "temp/"
   :format "pdf"
   :cmd "pdflatex"
   :extensions #{".bib" ".tex" ".bbl" ".aux"}
   :exclude  #"flycheck|\.\#|\~"})

(def test-config
  #_{:root "../tensor-pca/"
   :master "tensor-pca"}
  {:root "../inference/"
   :master "writeup"})

(defn new-system-map [config]
  (let [config (merge default-config config)]

    (assert (and (:root config) (:master config))
            "need to specify :root and :master in config")

    (component/system-map
     :latex-runner (new-latex-component config)
     :bibtex-runner (new-bibtex-component config)
     :latex-watchrun (new-watch-run
                      (filter (file-extension-map #{".tex" ".bbl" ".aux"}) )
                      (file-extension-cache-filter #{".bbl"})
                      (file-extension-cache-filter #{".aux"}))
     :bibtex-watchrun (new-watch-run
                       (filter (file-extension-map #{".bib" ".aux"}))
                       (file-extension-cache-filter #{".aux"}))
     :watcher (new-watcher-component config))))

(defn new-system-dependency-map []
  {:latex-runner {}
   :bibtex-runner {}
   :latex-watchrun {:latex :latex-runner}
   :bibtex-watchrun {:bibtex :bibtex-runner}
   :watcher {:latex :latex-watchrun
             :bibtex :bibtex-watchrun}})

(defn new-system [config]
  (component/system-using
   (new-system-map config)
   (new-system-dependency-map)))
