(ns latex-compile.latex.regexp)

;;;; some helpers

(defn re-find-named
  "find regex `re' in string `s' and return map with named groups according to `names'"
  [{:keys [re] :as names} s]
  (when-let [groups (re-find re s)]
    (->> groups
        (zipmap (map names
                     (range (count groups))))
        (#(dissoc % nil))
        )))

(defn map-vals [f m]
  "apply f to the values of the map m"
  (->> (for [[k v] (seq m)]
         [k (f v)])
       (into {})))

;;;; raw regular expressions for latex log files

(def latex-regexs-pre
  {:bad-pdftex-warning
   {:re #"^(.+)(pdfTeX warning.*)$"
    1 :before-split 2 :after-split}  ; can get rid of those

   :chapter
   {:re #"^Chapter (\d+)\." ; not sure what generates those (koma scrbook?)
    1 :chaper}

   :latex-warning-reference-full
   {:re #"^LaTeX Warning: Reference `(.+)' on page (\d+) undefined on input line (\d+)\."
    1 :label
    2 :page
    3 :line}

   :latex-warning-reference {:re #"^LaTeX Warning\: Reference "}

   :latex-error-undefined-control-sequence {:re #"^! Undefined control sequence\."}

   :package
   {:re #"^Package (\S+)"
    1 :package-name}

   :package-variant
   {:re #"^Package\: (\S+)"
    1 :package-name}

   :latex-info-redefining
   {:re #"LaTeX Info: Redefining (\S+) on input line (\d+)."
    1 :command
    2 :line}

   :page
   {:re #"(^\[|\s\[)(\d+)" ; matches "[2" and "bla [2"
    2 :page}

   :file-relative
   {:re #"(^|\s)\((\./[\S&&[^\)]]+)(\s|\)|$)" ; matches "(./abstract.tex)" or "(./abstract "
    2 :filename}

   :file-absolute
   {:re #"(^|\s)\((/[\S&&[^\)]]+)(\s|\)|$)" ; matches "(/book.tex)" or "(/book.tex "
    2 :filename}

   :latex-warning-undefined
   {:re #"^LaTeX Warning\: There were undefined references\."}

   :latex-warning-rerun
   {:re #"^LaTeX Warning\: Label\(s\) may have changed\. Rerun to get cross-references right\."}

   :latex-warning-biber-rerun
   {:re #"^Package biblatex Warning\: Please \(re\)run Biber"}

   :latex-warning-bibtex-rerun
   {:re #"^Package biblatex Warning\: Please \(re\)run BibTeX"}


   :line
   {:re #"^l\.(\d+) (.*$)" ; matches "l.34 \typo"
    1 :line
    2 :message}

   :empty-line {:re #"^\s*$"}

   :latex-error-too-many {:re #"^! Too many }'s\."}
   :latex-error-math-mode {:re  #"^! Missing \$ inserted" }

   :latex-error-paragraph-ended
   {:re #"^! Paragraph ended before (\S+) was complete\."
    1 :command}

   :latex-error-runaway {:re #"^Runaway argument\?"}

   :latex-error-package
   {:re #"^! LaTeX Error: File `(\S+)' not found\."
    1 :file}

   :latex-error-stop {:re #"^! Emergency stop\."}

   :latex-file-line-error {:re #"^(\S+):(\d+): (.*)$"
                           1 :file
                           2 :line
                           3 :error}

   :latex-error-fatal
   {:re #"^!  ==> Fatal error occurred, no output PDF file produced!"}})

(comment

  (re-matches #"^(\S+):(\d+): (.*)$" "./sections/introduction.tex:7: Undefined control sequence.")
  )

(defn latex-errors [x] ; non-fatal latex errors
  (some x  #{:latex-error-too-many
             :latex-error-runaway
             :latex-error-math-mode
             :latex-error-paragraph-ended
             :latex-error-package
             :latex-file-line-error
             :latex-error-undefined-control-sequence}))

(defn latex-warning-rerun [x]
  (some x #{:latex-warning-rerun}))

(defn latex-fatal-errors [x]
  (or (some x #{:latex-error-fatal
                :latex-error-stop})
      (some->> x
               :latex-file-line-error
               :error
               (re-find #"Fatal"))))

(comment
  (latex-fatal-errors {:latex-file-line-error {:error "==> Fatal error"}})
)

;;; compile the raw regular expressions

(def latex-regexs
  (->> (for [[k v] (seq latex-regexs-pre)]
         [k (partial re-find-named v)])
       (into {})))

(def important-log-entries
  #{:latex-error-fatal
    :latex-error-stop
    :latex-error-too-many
    :latex-error-runaway
    :latex-error-math-mode
    :latex-error-paragraph-ended
    :latex-error-package
    :latex-error-undefined-control-sequence})

(def location-log-entries-pre
  #{:file-relative :file-absolute :page :chapter})

(def location-log-entries-post
  #{:line})
