(ns latex-compile.latex
  (:require [clojure.java.shell :as sh]
            [latex-compile.helpers :as helpers :refer [graph-run
                                                       opts->args
                                                       opts->map
                                                       opts-update
                                                       path]]
            [latex-compile.latex.log :as latex-log]
            [latex-compile.latex.regexp :as latex-log-regex]
            [me.raynes.fs :as fs]
            [plumbing.core :as plumbing :refer [fnk]]
            [text-decoration.core :as deco]))

(def latex-opts-graph
  {:latex-opts-static
   (fnk []
        [[:interaction "batchmode"]
         [:synctex "1"]
         [:file-line-error]
         [:src-specials]])
   :latex-opts
   (fnk [latex-opts-static temp-dir file format]
        ;; FIXME: using just the name of the file could create bad conflicts
        (conj latex-opts-static
              [:output-directory (->> (str temp-dir (fs/name file))
                                     (fs/file)
                                     (fs/normalized)
                                     (path))]
              [:output-format format]
              [:jobname (fs/name file)]))})

(defn trace-fn [f]
  f
  #_(fn [& args]
    (prn args)
    (apply f args)))

(def latex-graph
  "fnk's for latex compilation; open keys: file, latex-opts, and cmd"
  {:latex-opts-map
   (fnk [latex-opts]
        (-> latex-opts
            opts->map))
   :start-time
   (fnk [] (System/nanoTime))
   :stop-time
   (fnk [latex-run] (System/nanoTime))
   :latex-run-duration
   (fnk [start-time stop-time] (/ (- stop-time start-time ) 1000000.0))
   :assert-out-dir
   (fnk [latex-opts-map file]
        (assert (:output-directory latex-opts-map)
                (format "No :output-directory in latex options specified (file: %s)" file)))
   :latex-out-dir
   (fnk [latex-opts-map assert-out-dir]
        (-> latex-opts-map
            :output-directory
            fs/normalized
            path))
   :latex-work-dir
   (fnk [file]
        (-> file
            fs/parent
            fs/normalized
            path))
   :file-relative
   (fnk [file]
        (-> file
            fs/base-name))
   :latex-opts-normalized
   (fnk [latex-out-dir latex-opts latex-work-dir]
        (opts-update latex-opts
                     :output-directory (helpers/->relative-path
                                        latex-work-dir
                                        latex-out-dir)))
   :latex-args
   (fnk [latex-opts-normalized]
        (opts->args latex-opts-normalized))
   :latex-partial
   (fnk [cmd latex-args]
        (apply partial (trace-fn sh/sh) cmd latex-args))
   :jobname
   (fnk [latex-opts-map]
        (:jobname latex-opts-map))
   :latex-output-files
   (fnk [latex-out-dir jobname]
        (plumbing/map-from-keys
           #(str latex-out-dir "/" jobname "." (name %))
           [:log :pdf]))
   :latex-run-mkdirs
   (fnk [latex-out-dir]
        (fs/mkdirs latex-out-dir))
   :latex-run
   (fnk [latex-work-dir latex-partial latex-run-mkdirs file-relative start-time]
        (latex-partial file-relative
                       :dir latex-work-dir
                       ;; avoid line breaks in latex log?
                       :env (into {"max_print_line" "1048"
                                   "error_line" "254"
                                   "half_error_line" "238"}
                                  (System/getenv))))
   :latex-run-print
   (fnk [latex-run-duration file-relative status]
        (helpers/printfln-deco
         [:bold :cyan]
         "[latex] file: %s, duration: %s, status: %s"
         file-relative latex-run-duration status))
   :latex-run-error
   (fnk [latex-run]
        ;; check if latex exited normally
        (when (pos? (:exit latex-run))
          (helpers/printfln-deco
           [:bold :red]
           "[latex-error] %s"
           (:err latex-run))))
   :latex-log
   (fnk [latex-output-files latex-run]
        (when (fs/exists? (:log latex-output-files))
          (-> latex-output-files
              :log
              slurp ; FIXME: check if this file exists
              latex-log/parse)))
   :latex-line-errors
   (fnk [latex-log]
        (->> (keep :latex-file-line-error latex-log)
             (map (juxt :error :file :line))
             (map (partial apply format
                           "error: %s file: %s line: %s"))))
   :latex-line-errors-print
   (fnk [latex-line-errors]
        (->> latex-line-errors
             (map (partial str "[latex-log] ") )
             (map (partial helpers/printfln-deco [:red]))
             (dorun)))
   :ensure-output (fnk [latex-run-error
                        latex-line-errors-print
                        latex-run-print])
   :output-file
   (fnk [latex-output-files latex-opts-map status ensure-output]
        (when-not (= status :fatal)
          (-> latex-opts-map
              (:output-format)
              (keyword)
              (latex-output-files))))
   :status
   (fnk [latex-log]
        (condp some latex-log
          latex-log-regex/latex-warning-rerun :rerun
          latex-log-regex/latex-fatal-errors :fatal
          latex-log-regex/latex-errors :error
          :success ))})

(defn latex [{:keys [file] :as m}]
  (loop []
    (let [latex-run (graph-run (merge latex-graph latex-opts-graph) m
                               :keep-key? #{:output-file :status :file-relative})]
      (if (= :rerun (:status latex-run))
        (do (println
             (format "[latex] rerun compilation (file: %s)"
                     (:file-relative latex-run)))
            (recur))
        (dissoc latex-run :status :file-relative)))))

(defn latex-single [{:keys [file] :as m}]
  (dissoc
   (graph-run (merge latex-graph latex-opts-graph) m
              :keep-key? #{:output-file :status :file-relative})
   :status :file-relative))
