(ns load-system.core
  (:require [duct.util.system :as d]
            [meta-merge.core :refer [meta-merge]]
            [clojure.java.io :as io]
            [clojure.data :as cd]
            [clojure.pprint :refer [pprint]])
  (:import java.io.File))



(defn source
  "return variant v source s if it exists (e.g., :file or :resource)"
  [[v s]]
  (case v
    :resource (io/resource s)
    :file (let [f ^File (io/file s)]
            (when (.exists f)
              f))))

(defn keep-source
  "return list of sources"
  [sources]
  (keep source sources))



(defn system-diff
  "output difference of systems a and b"
  [a b & [ks]]
  (->>
   (cd/diff a b)
   (take 2)
   (zipmap (or ks [:only-in-a :only-in-b]))))

(defn print-system-diff
  "print difference of system a and b"
  [a b & [ks]]
  (pprint (system-diff a b ks)))



(defn- get-config
  "generate config from sources and bindings"
  [sources bindings]
  (->> sources
       (map #(d/read-config % bindings))
       (apply meta-merge)))

(defn- configure-endpoints
  "configure endpoints of system according to config"
  [system {:keys [endpoints config]}]
  (->> endpoints
       (keys)
       (select-keys config)
       (meta-merge system)))

(defn load-system
  "load system from sources and bindings with configured endpoints"
  [sources bindings]
  (-> (d/load-system sources bindings)
      (configure-endpoints (get-config sources bindings))))
