(ns mathdoc.cljs.svg
  (:require [goog.string :as gstring]
            [goog.string.format]
            [cuerdas.core :as str]))

(defn path-d
  "convert args to svg path format"
  [& args]
  (some->>
   args
   (map #(cond-> % (keyword? %) (name)))
   (flatten)
   (str/join " ")))


(defn circle [r]
  [:circle
   {:cx 0
    :cy 0
    :r r}])

(defn point [m r [x y]]
  [:circle
   (merge m
          {:cx x
           :r  r
           :cy y})])

(defn line [m #_{:keys [points]} & children]
  [:polyline
   (-> m
       (update :points #(apply path-d (apply concat %))))
   #_{:points (apply path-d (apply concat points))}
   (map-indexed (fn [i x] (vary-meta x assoc :key i)) children)])

(defn arc [r [x1 y1] [x2 y2]]
  [:path {:d (path-d "M" x1 y1 "A" r r 0 0 0 x2 y2)}])

(defn angle [r [x1 y1] [x2 y2]]
  [:path {:d (path-d "M" x1 y1 "A" r r 0 0 0 x2 y2 "L" 0 0 "Z")}])

(defn text [[x y] s]
  [:text {:x x :y y}
   s])



(defn arrow [[x1 y1] [x2 y2]]
  [:path {:d (path-d :M x1 y1 :L x2 y2)
          :marker-end "url(#Triangle)"}])


(defn origin-svg [{:keys [x-scale y-scale padding]} & children]
  [:svg {:viewBox (str/format
                   "$x $y $width $height"
                   {:x      (- 0 padding x-scale)
                    :y      (- 0 padding y-scale)
                    :width  (+ (* 2 x-scale) (* 2 padding))
                    :height (+ (* 2 y-scale) (* 2 padding))})}
   (map-indexed
    (fn [i c]
      (vary-meta c assoc :key i))
    children)])
