(ns mathdoc.core
  (:require [integrant.core :as ig]
            [clojure.spec :as s]
            [taoensso.tufte :as tufte]
            [clojure.spec :as s]
            [duct.core.protocols :as dcp]
            [duct.core :as duct]
            [clojure.java.io :as io]
            [clojure.string :as str]))

;;; specs

(s/def ::logger
  #(satisfies? dcp/Logger %))
(s/def ::root string?)
(s/def ::port int?)

(s/def ::path
  (s/and string?
         not-empty
         #(not (str/ends-with? % "/"))))

;;; derivations

(defn derive-all [derivations]
  (doseq [[tag parent] derivations]
    (derive tag parent)))

;;; implementation

(defmethod ig/init-key
  ::port
  [_ port]
  (s/assert ::port port))

(defmethod ig/init-key
  ::root
  [_ root]
  (s/assert ::root root))

;;; string format

(defn load-var [sym]
  (require (symbol (namespace sym)))
  (find-var sym))

#_ (load-var 'prone.middleware/wrap-exceptions)

(defmethod ig/init-key
  ::var
  [_ s]
  (s/assert symbol? s)
  (load-var s))

(defmethod ig/init-key
  ::get-in
  [_ [m ks & [not-found]]]
  (s/assert map? m)
  (if not-found
    (get-in m ks not-found)
    (get-in m ks)))

(defmethod ig/resume-key ::resume-identity [_ _ _ v] v)

(defmethod ig/init-key
  ::join-paths
  [_ [& paths]]
  (->> paths
       (s/assert (s/coll-of ::path))
       (apply io/file)
       (str)))

(defmethod ig/init-key
  ::path
  [_ path]
  (s/assert ::path path))

(defmethod ig/init-key
  ::string
  [_ s]
  (s/assert string? s))

(defmethod ig/init-key
  ::format
  [_ args]
  (apply format args))

(defmethod ig/init-key
  ::parent-dir
  [_ f]
  (s/assert ::path f)
  (str (.getParentFile (io/file f)))
  #_(as-> (str content-dir "/" file) _
    (str/split _  #"/")
    (drop-last _)
    (str/join "/" _)
    (str _ "/_metadata.yaml")))

(comment

  (ig/init-key ::parent-dir "/test/file")

  )


#_(ig/init-key [::test ::format] ["%s--%s--%s" 1 2 3])

;;; config sources

(s/def ::sources (s/coll-of keyword?))

(defmethod ig/init-key
  ::config-sources
  [_ sources]
  (s/assert ::sources sources)
  sources)

;; config

(defn read-config [& sources]
  (->> sources
       (flatten)
       (partition 2)
       (map
        (fn [[variant source]]
          (case variant
            :resource (io/resource source)
            :file (io/file source))))
       (apply duct/read-config)))

(defn exec
  [config]
  (let [system (-> config duct/prep ig/init)]
    (duct/add-shutdown-hook ::exec #(ig/halt! system))
    (.. Thread currentThread join)))

;; profiling

(defn profile-init-key [k v]
  (tufte/p k (#'ig/init-key k v)))

(defmethod ig/init-key
  ::profile?
  [_ v]
  (s/assert boolean? v))

(defn profile-init
  [{:keys [::profile?] :as config}]
  (ig/build config (keys config) profile-init-key))


;;; tests

(comment

  (read-config
   :resource "mathdoc/core/config.edn"
   :resource "mathdoc/server/config.edn"
   :resource "mathdoc/compiler/config.edn")

  )
