(ns mathdoc.process.transform.pause
  (:require [clojure
             [spec :as s]
             [walk :as walk]]
            [clojure.core.match :as m]
            [taoensso.timbre :as log]
            [integrant.core :as ig]
            [mathdoc.specs.process :as msp]
            [duct.core :as duct]))

;;; specs

(s/fdef process
        :args (s/cat :data (s/keys :req-un [::msp/ast]))
        :ret ::msp/ast)

;;; implementation

(defn skip
  [skip-pred f]
  (fn [v]
    (if (skip-pred v)
      v
      (f v))))

(def PAUSE
  {:type :Para,
   :content
   {:elems
    [{:type :Str, :content "."}
     {:type :Space}
     {:type :Str, :content "."}
     {:type :Space}
     {:type :Str, :content "."}]}})

(defn partition-with
  [pred coll]
  (->>
   coll
   (reduce
    (fn [acc x]
      (if (pred x)
        (conj acc [x])
        (update acc (dec (count acc)) conj x)))
    [[]])))

(defn not-contain-pause
  [v]
  (not (and (vector? v)
            (every? map? v)
            (some #{PAUSE} v))))

(defn wrap-div [attr elems]
  {:type :Div
   :content
   {:attr attr
    :elems elems}})

(defn process-pause
  [v]
  (->> v
       (partition-with #{PAUSE})
       (map (fn [x]
              (m/match
               [x]
               [[(_ :guard #{PAUSE}) & xs]]
               [(wrap-div {:class ["fragment"]} (vec xs))]
               :else x)))
       (apply concat)
       (vec)))

(defn process-data [input]
  (walk/postwalk
   (skip not-contain-pause process-pause)
   input))

(defn process [{:keys [logger ast]}]
  (duct/log logger :info :process)
  (process-data ast))


;;; integrant

(defmethod ig/init-key
  :mathdoc.process.transform/pause
  [_ v]
  (process v))

