(ns mathdoc.core
  (:require [clojure
             [spec :as s]
             [string :as str]]
            [clojure.java.io :as io]
            [duct.core :as duct]
            [duct.core.protocols :as dcp]
            [integrant.core :as ig]
            [taoensso.tufte :as tufte])
  (:import java.net.URI))

;;; specs

(s/def ::logger #(satisfies? dcp/Logger %))
(s/def ::root string?)
(s/def ::port int?)

(s/def ::base-uri
  (s/and string?
         #(str/starts-with? % "/")
         #(str/ends-with? % "/")))

(s/def ::path
  (s/and string?
         not-empty
         #(not (str/ends-with? % "/"))))

;;; derivations

(defn derive-all [derivations]
  (doseq [[tag parent] derivations]
    (derive tag parent)))

;;; implementation

(defmethod ig/init-key
  ::port
  [_ port]
  (s/assert ::port port))

(defmethod ig/init-key
  ::uri-to-dir
  [_ uri]
  (-> uri
      (str/replace #"^/" "")
      (str/replace #"/$" "")))

(defmethod ig/init-key
  ::base-uri
  [_ base-uri]
  (s/assert ::base-uri base-uri))

(defmethod ig/init-key
  ::root
  [_ root]
  (s/assert ::root root))

;;; string format

(defn load-var [sym]
  (require (symbol (namespace sym)))
  (find-var sym))

(defmethod ig/init-key
  ::var
  [_ s]
  (s/assert symbol? s)
  (load-var s))

(defmethod ig/init-key
  ::get-in
  [_ [m ks & [not-found]]]
  (s/assert map? m)
  (get-in m ks not-found))

(defmethod ig/resume-key ::resume-identity [_ _ _ v] v)

(defmethod ig/suspend-key! ::resume-identity [_ _])

(defmethod ig/init-key
  ::join-paths
  [_ paths]
  (->> paths
       (s/assert
        (s/coll-of
         (s/or :non-empty ::path
               :empty #{""})))
       (apply io/file)
       (str)))

(defmethod ig/init-key
  ::path
  [_ path]
  (s/assert ::path path))

(defmethod ig/init-key
  ::string
  [_ s]
  (s/assert string? s))

(defmethod ig/init-key
  ::coll-of-string
  [_ coll]
  (s/assert (s/coll-of string?) coll))

(defmethod ig/init-key
  ::format
  [_ args]
  (apply format args))

(defmethod ig/init-key
  ::parent-dir
  [_ f]
  (s/assert ::path f)
  (str (.getParentFile (io/file f))))

;;; config sources

(s/def ::sources (s/coll-of keyword?))

(defmethod ig/init-key
  ::config-sources
  [_ sources]
  (s/assert ::sources sources)
  sources)

;; config

(defn read-config [& sources]
  (->> sources
       (flatten)
       (partition 2)
       (map
        (fn [[variant source]]
          (case variant
            :resource (io/resource source)
            :file (io/file source))))
       (apply duct/read-config)))

(defn exec
  [config]
  (let [system (-> config duct/prep ig/init)]
    (duct/add-shutdown-hook ::exec #(ig/halt! system))
    (.. Thread currentThread join)))

;; profiling

(defn profile-init-key [k v]
  (tufte/p k (#'ig/init-key k v)))

(defmethod ig/init-key
  ::profile?
  [_ v]
  (s/assert boolean? v))

(defn profile-init
  [{:keys [::profile?] :as config}]
  (ig/build config (keys config) profile-init-key))


(defn relative-path
  "Path of `child` relative to `parent` as string. Returns `nil` if `child` is not a child of `parent`"
  [parent child]
  (let [child-uri ^URI (.toURI (io/file child))
        parent-uri ^URI (.toURI (io/file parent))
        relative-child-uri ^URI (.relativize parent-uri child-uri)]
    (when-not (= child-uri relative-child-uri)
      (.getPath relative-child-uri))))


;;; tests

(comment

  (read-config
   :resource "mathdoc/core/config.edn"
   :resource "mathdoc/server/config.edn"
   :resource "mathdoc/compiler/config.edn")

  (ig/init-key ::uri-to-dir "/uri")

  (relative-path "a/" "a/b c/d")

  (ig/init-key ::parent-dir "/test/file")

  )


#_(ig/init-key [::test ::format] ["%s--%s--%s" 1 2 3])

#_ (load-var 'prone.middleware/wrap-exceptions)
