(ns mathdoc.cljs.loadcanvas
  (:require
   [taoensso.timbre :as log]
   [integrant.core :as ig]
   [cljs.core.async :as a]
   [dommy.core :as dommy :refer-macros [sel sel1]]
   [mathdoc.cljs.reveal :as reveal]
   [mathdoc.cljs.util :as util])
  (:require-macros
   [cljs.core.async.macros :refer [alt! go go-loop]]
   [cljs.spec.alpha :as s]))

(defn clear-board []
  (.clearMemory js/Board))

(defn make-image [src]
  (doto (new js/Image)
    (aset "src" src)))

(defn draw-image [canvas img]
  ;; scale so that height matches
  ;; center the image horizontally
  (let [#_#_canvas (sel1 "#board")
        ratio (/ (.-height canvas) (.-height img))
        shift (* 0.5 (- (.-width canvas) (* ratio (.-width img))))]
    (->
     canvas
     (.getContext "2d")
     (.drawImage img
                 0 0 (.-width img) (.-height img)
                 shift 0 (* ratio (.-width img)) (* ratio (.-height img))))))

(defn load-image [canvas src]
  (let [img (make-image src)]
    (-> img
        (aset "onload"
              #(do
                 (clear-board)
                 (draw-image canvas img))))))

(defn load-elem-image [canvas elem]
  (when-let [src (dommy/attr elem "data-canvas-src")]
    (load-image canvas src)
    #_(let [board (sel1 "#board")]
        (go
          (.log js/console "load")
          (dommy/remove-style! board "opacity")
          (dommy/add-class! board "fadedout")
          (a/<! (a/timeout 100))
          (dommy/remove-class! board "fadedout")))))

;;; integrant

(def config
  {::listen       {:fragmentshown-tap (ig/ref ::reveal/fragmentshown)
                   :slidechanged-tap  (ig/ref ::reveal/slidechanged)
                   :canvas-delay      (ig/ref ::canvas-delay)}
   ::canvas-delay nil
   ::trash        {:ready (ig/ref ::reveal/ready)}})

(defmethod ig/init-key
  ::canvas-delay
  [_ _]
  (log/info ::canvas-delay)
  (delay (sel1 "#board")))

(defmethod ig/init-key
  ::trash
  [_ {:keys [ready]}]
  (go
    (a/<! ready)
    (log/info ::trash)
    (dommy/listen!
     (sel1 "#trash")
     :click clear-board))
  ::trash)

(defmethod ig/resume-key
  ::trash
  [_ _ _ v]
  (log/info ::trash :resume)
  v)

(defmethod ig/init-key
  ::listen
  [_ {:keys [canvas-delay slidechanged-tap fragmentshown-tap]}]
  (log/info ::listen :init)

  (s/assert fn? slidechanged-tap)
  (doto (a/chan (a/dropping-buffer 10))
    (slidechanged-tap)
    (util/consume-chan
     (fn canvas-image-slide [ev]
       (log/info ::slidechanged)
       (clear-board)
       (load-elem-image @canvas-delay (aget ev "currentSlide")))))

  (s/assert fn? fragmentshown-tap)
  (doto (a/chan (a/dropping-buffer 10))
    (fragmentshown-tap)
    (util/consume-chan
     (fn canvas-image-fragment [ev]
       (log/info ::fragmentshown)
       (doseq [f (aget ev "fragments")]
         (load-elem-image @canvas-delay f)))))

  ::listen)

(defmethod ig/resume-key
  ::listen
  [_ _ _ v]
  (log/info ::listen :resume)
  v)

(comment

  (make-image "lecture1-title.png")

  (load-image "lecture1-title.png")

  (dommy/remove-style!
     (sel1 "#board")
     "opacity")

  (dommy/remove-class!
     (sel1 "#board")
     "fadedout")

  (dommy/add-class!
     (sel1 "#board")
     "fadedout")

  (dommy/toggle-class!
     (sel1 "#board")
     "fadedout")
)
