(ns mathdoc.process
  (:require [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [duct.core :as duct]
            [integrant.core :as ig]
            [mathdoc.core :as core]
            [medley.core :as med]
            [meta-merge.core :as m]
            [taoensso.tufte :as tufte]
            [clojure.string :as str])
  (:import java.net.URI))



;; specs

(s/def ::file string?)
(s/def ::root string?)
(s/def ::content-dir string?)
(s/def ::prepend-files (s/coll-of string?))
(s/def ::append-files (s/coll-of string?))
(s/def ::target-dir string?)

(s/def ::inputfiles
  (s/keys :req-un [::file
                   ::content-dir
                   ::root
                   ::prepend-files
                   ::append-files]))

(s/def ::targetfile
  (s/keys :req-un [::file
                   ::root
                   ::target-dir]))



;; implementation

(defmethod ig/init-key
  ::root
  [_ v]
  (s/assert ::root v)
  v)

(defmethod ig/init-key
  ::file
  [_ v]
  (s/assert ::file v)
  v)



(defn process-run
  [config file]
  (s/assert map? config)
  (s/assert string? file)
  (->
   (m/meta-merge
   config
   {:mathdoc.process/file file})
   (duct/prep)
   (core/profile-init)
   :mathdoc.process.fragment/copy))

(defn set-namespace
  [ns k]
  (s/assert string? ns)
  (s/assert keyword? k)
  (keyword ns (name k)))

(s/def ::options map?)
(s/def ::environment #{:production :development})

(s/def :mathdoc/process
  (s/keys
   :opt-un [::environment ::core/profile? ::options]
   :req-un [::root]))

(defn get-process-config
  [{:keys [options root environment] :as config}]
  (s/assert :mathdoc/process config)
  (m/meta-merge
   (duct/read-config
    (io/resource "mathdoc/process/config.edn"))
   (med/map-keys (partial set-namespace "mathdoc.process.options") options)
   (when environment
     {:duct.core/environment environment})
   {:mathdoc.process/root root}))

(defmethod ig/init-key
  :mathdoc/process
  [_ {:keys [logger profile?] :as config}]
  {:pre [logger]
   :post [(s/assert fn? %)]}
  (s/assert :mathdoc/process config)
  (let [process-config (get-process-config config)]
    (fn process-files [files]
      (s/assert (s/coll-of string?) files)
      (duct/log logger :info :process-files (vec files))
      (tufte/add-basic-println-handler! {})
      (tufte/profile
       {:when profile?}
       (->> files
            (map #(process-run process-config %))
            doall)))))

(defn directory? [d]
  (.isDirectory (io/file d)))

(defn file-exists? [f]
  (.exists (io/file f)))

#_(directory? "../talks/content")

(defn content-files [content-path]
  (->> (io/file content-path)
       file-seq
       (filter file-exists?)
       (remove directory?)
       (map (partial core/relative-path content-path))
       (filter #(str/ends-with? % ".md"))))

(comment

  (content-files "../talks/content")

  )

(derive ::compiler :duct/compiler)

(defmethod ig/init-key
  ::compiler
  [_ {:keys [content-path process-fn]}]
  (process-fn (content-files content-path)))

#_(defmethod ig/resume-key ::compiler [_ _ _ v] v)

(derive ::compiler :mathdoc.core/resume-identity)

(comment
  (ig/init-key :mathdoc/process
               {:root    "test-root"
                :profile? false
                :environment :development
                :options {:content-dir "test"}})

  (ig/init-key :mathdoc/process
               {:root    "../../talks"
                :profile? false
                :environment :development
                :options {:content-dir "content-slides"
                          :target-dir    "html-slides"
                          :include-dir   "html-slides"
                          :prepend-files ["content-slides/latexmacros.tex"]
                          :append-files  ["content-slides/metadata.yaml"]
                          :template      "html-slides/template.html"
                          :commitscript  "scripts/commit-generated.sh"}})

  )
