(ns mathdoc.process.conform
  (:require [clojure.spec.alpha :as s]
            [medley.core :as med]
            [taoensso.timbre :as log]
            [integrant.core :as ig]
            [mathdoc.specs.process :as msp]
            [duct.core :as duct]))

;;; process specs

(s/fdef process
        :args (s/cat
               :data (s/keys :req-un [::msp/jsonedn]))
        :ret ::msp/ast)



;;; ast specs

(defmulti elem-type :type)

(defmethod elem-type
  :default [_]
  (s/conformer
   (fn default-codeblock-conformer [x]
     {:type :CodeBlock
      :content {:attr {:id nil
                       :class ["nonconformingmarkdownerror" "clojure"]
                       :data-map {}}
                :code (prn-str x)}})))

(defmethod elem-type
  :Str [_]
  (s/keys :req-un [:string/content]))

(defmethod elem-type
  :Note [_]
  (s/keys :req-un [:para/content]))

(defmethod elem-type
  :CodeBlock [_]
  (s/keys :req-un [:codeblock/content]))


(defmethod elem-type
  :Emph [_]
  (s/keys :req-un [:para/content]))

(defmethod elem-type
  :Subscript [_]
  (s/keys :req-un [:para/content]))


(defmethod elem-type
  :Plain [_]
  (s/keys :req-un [:para/content]))

(defmethod elem-type
  :RawInline [_]
  (s/keys :req-un [:raw/content]))

(defmethod elem-type
  :RawBlock [_]
  (s/keys :req-un [:raw/content]))

(s/def :raw/content
  (s/cat
   :format string?
   :rawstring string?))

(defmethod elem-type
  :Strong [_]
  (s/keys :req-un [:para/content]))

(defmethod elem-type
  :BulletList [_]
  (s/keys :req-un [:list/content]))

(defmethod elem-type
  :OrderedList [_]
  (s/keys :req-un [:orderedlist/content]))

(defmethod elem-type
  :Quoted [_]
  (s/keys :req-un [:quote/content]))

(defmethod elem-type
  :Math [_]
  (s/keys :req-un [:math/content]))

(s/def :math/content
  (s/cat :math-type (s/or :inline #{{:type :InlineMath}}
                          :display #{{:type :DisplayMath}})
         :math-content string?))

(defmethod elem-type
  :Header  [_]
  (s/keys :req-un [:header/content]))

(defmethod elem-type
  :Div [_]
  (s/keys :req-un [:div/content]))

(defmethod elem-type
  :Span [_]
  (s/keys :req-un [:div/content]))

(defmethod elem-type
  :Para [_]
  (s/keys :req-un [:para/content]))

(defmethod elem-type
  :Table [_]
  (s/keys :req-un [:table/content]))

(defmethod elem-type
  :HorizontalRule [_]
  any?)

(defmethod elem-type
  :Space [_]
  any?)

(defmethod elem-type
  :LineBreak [_]
  any?)

(defmethod elem-type
  :BlockQuote [_]
  (s/keys :req-un [:para/content]))

(defmethod elem-type
  :Cite [_]
  (s/keys :req-un [:cite/content]))

(defmethod elem-type
  :Link [_]
  (s/keys :req-un [:link/content]))

(defmethod elem-type
  :Image [_]
  (s/keys :req-un [:image/content]))

(defmethod elem-type :SoftBreak [_]
  any?)

(s/def ::elems
  (s/* ::elem))

(s/def ::elem
  (s/multi-spec elem-type :type))

(s/def ::attr
  (s/cat
   :id string?
   :class (s/coll-of string?)
   :data-map (s/and
              (s/coll-of (s/tuple string? string?))
              (s/conformer (partial into {}))
              (s/conformer (partial med/map-keys keyword)))))

(s/def :quote/content
  (s/cat :quote-type (s/or :double #{{:type :DoubleQuote}}
                           :single #{{:type :SingleQuote}})
         :elems (s/spec ::elems)))

(s/def :string/content
  string?)

(s/def :header/content
  (s/cat
   :level int?
   :attr (s/spec ::attr)
   :elems (s/spec ::elems)))

(s/def :para/content
  (s/cat
   :elems ::elems))

(s/def :list/content
  (s/cat
   ;; probably not the right way to spec list items
   :items (s/* (s/coll-of ::elem))))

(s/def :orderedlist/content
  (s/cat
   ;; probably not the right way to spec list items
   :enum-type any? #_(s/cat :start int?)
   :items (s/coll-of (s/coll-of ::elem))))

(s/def :div/content
  (s/cat
   :attr (s/spec ::attr)
   :elems (s/spec ::elems)))


(s/def :codeblock/content
  (s/cat
   :attr (s/spec ::attr)
   :code string?))



(s/def :table/content
  (s/cat :caption (s/spec ::elems)
         :align any?
         :numbers any?
         :thead (s/coll-of ::elems)
         :tbody (s/coll-of (s/coll-of ::elems))))

(s/def :cite/content
  (s/cat
   :citations (s/coll-of
               (s/map-of
                keyword? any?))
   :elems (s/spec ::elems)))

(s/def :link/content
  (s/cat
   :attr (s/spec ::attr)
   :elems (s/spec ::elems)
   :target (s/spec (s/cat
                    :href string?
                    :something string?))))

(s/def :image/content
  (s/cat
   :attr (s/spec ::attr)
   :elems (s/spec ::elems)
   :target (s/spec (s/cat
                    :href string?
                    :something string?))))

(s/def ::blocks
  ::elems)

(s/def ::data
  (s/keys :req-un [::blocks ::meta]))



;;; implementation

(defn get-conform [d]
  (as-> (s/conform
         ::data d) _
        (if-not (s/invalid? _)
          _
          (throw
           (ex-info
            "failed to conform to specification"
            (s/explain-data ::data d))))))

(defn process [{:keys [logger jsonedn]}]
  (duct/log logger :info :process)
  (get-conform jsonedn))

;;; integrant

(defmethod ig/init-key
  :mathdoc.process/conform
  [_ v]
  (process v))
