(ns mathdoc.process.write.html
  (:require [clojure.string :as str]
            [hiccup.core :as h]
            [integrant.core :as ig]
            [taoensso.timbre :as log]
            [clojure.walk :as walk]
            [duct.core :as duct]))

(defmulti html :type)

(defn html-elems [elems]
  (->> elems
       (map html)))

(defmethod html
  :Str [{s :content}]
  s)

(defmethod html
  :RawInline [{{:keys [format rawstring]} :content}]
  #_(log/error ::rawinline)
  (if (= "html" format)
    (str #_"<!-- begin raw html -->"
         rawstring
         #_"<!-- end raw html -->")
    ""))

(defmethod html
  :RawBlock [{{:keys [format rawstring]} :content}]
  #_(log/error ::rawblock)
  (if (= "html" format)
    (str #_"\n<!-- begin raw html -->"
         rawstring
         #_"<!-- end raw html -->\n")
    ""))

(defmethod html
  :Space [_]  " ")

(defmethod html
  :LineBreak [_]  [:br] #_"<br>")

(defmethod html
  :HorizontalRule [_]
  ;; very bad hack
  "</section><section class=\"horizonal-rule\"")

(defmethod html
  :Table
  [{{:keys [caption thead tbody]} :content}]
  [:table
   [:caption (html-elems caption)]
   [:thead
    [:tr
     (for [h thead]
       [:th (html-elems h)])]]
   [:tbody
    (for [r tbody]
      [:tr
       (for [d r]
         [:td (html-elems d)])])]])

(defmethod html
  :Para
  [{{:keys [elems]} :content}]
  [:p (html-elems elems)])

(defmethod html
  :Note
  [{{:keys [elems]} :content}]
  [:div.sidenote (html-elems elems)])

(defmethod html
  :Plain
  [{{:keys [elems]} :content}]
  (html-elems elems))

(defmethod html
  :Header [{{:keys [level attr elems]} :content}]
  [(keyword (str "h" level))
   (html-elems elems)]
  #_(as->
     [(keyword (str "h" level))
      (html-elems elems)]
     _
     (if (= 1 level) [:section.titleslide _] _)))

(defmethod html
  :BulletList [{{:keys [:items]} :content}]
  [:ul
   (->>
    items
    (map html-elems)
    (map (partial into [:li])))])

(defmethod html
  :OrderedList [{{:keys [items enum-type]} :content}]
  [:ol {:start (or (some-> enum-type
                           seq
                           first) 1)}
   (->>
    items
    (map html-elems)
    (map (partial into [:li])))])

(defmethod html
  :BlockQuote
  [{{:keys [elems]} :content}]
  [:blockquote
   (html-elems elems)])

#_(html {:type :Div :content {:elems [] :attr {:id "refs"}}})

(defn html-attr [{:keys [id class data-map]}]
  (cond-> data-map
    (seq id) (assoc :id id)
    (seq class) (assoc :class (str/join " " class))))

(defmethod html
  :CodeBlock [{{:keys [code attr]} :content}]
  #_(log/info :code attr)
  [:pre
   [:code (html-attr attr)
    code]])

(defmethod html
  :Div
  [{{elems :elems {:keys [id class data-map] :as attr} :attr} :content}]
  #_(log/info class)
  (when-not (some #{"noexport"} class)
    (let [content (html-elems elems)
          #_      (if (some #{"notes"} class)
                    (list notes-style (html-elems elems))
                    (html-elems elems))
          attr    (html-attr attr)]
      (cond
        (:data-divtag data-map)      [(keyword (:data-divtag data-map))
                                      attr content]
        (:presection data-map)       content
        (= "refs" id)                [:section attr content]
        (#{2 3} (:section data-map)) [:section attr content]
        (= 1 (:section data-map))    content
        true                         [:div attr content]))))

(defmethod html
  :Span
  [{{:keys [elems attr]} :content}]
  [:span
   (html-attr attr)
   (html-elems elems)])

(defmethod html
  :Emph
  [{{:keys [elems]} :content}]
  [:em
   (html-elems elems)])

(defmethod html
  :Subscript
  [{{:keys [elems]} :content}]
  [:sub
   (html-elems elems)])

(defmethod html
  :Strong
  [{{:keys [elems]} :content}]
  [:strong
   (html-elems elems)])

(defn math-escape [x]
  (str/replace x #"[<>]" #(case %
                            ">" "\\gt"
                            "<" "\\lt")))

#_(math-escape "><")

(defmethod html
  :Math
  [{{:keys [math-content math-type]} :content}]
  (case (first math-type)
    :inline  [:span.math.inline (str "\\(" (math-escape math-content) "\\)")]
    :display [:div.math.display (str "\\[" (math-escape math-content) "\\]")]))

(defmethod html
  :Quoted
  [{{:keys [elems quote-type]} :content}]
  (case (first quote-type)
    :double [:span.quote.double
             "“"
             (html-elems elems)
             "”"]
    :single [:span.quote.single
             "‘"
             (html-elems elems)
             "’"]))

(defmethod html
  :Cite [{{:keys [citations elems]} :content}]
  (html-elems elems))

(defmethod html
  :Link [{{:keys [attr elems target]} :content}]
  [:a (merge (html-attr attr) {:href (:href target)})
   (html-elems elems)])

(defmethod html
  :Image [{{:keys [attr elems target]} :content}]
  [:img (merge (html-attr attr) {:src (:href target)})
   #_(html-elems elems)])

(defmethod html
  :SoftBreak [_] "\n")

(defn process [{:keys [input logger]}]
  (assert logger)
  (duct/log logger :info :process)
  (html-elems
   (:blocks
    input)))

;;; integrant

(defmethod ig/init-key
  :mathdoc.process.write/html
  [_ v]
  (process v))

(defmethod ig/init-key
  :mathdoc.process.write.html/string
  [_ {:keys [input logger]}]
  (duct/log logger :info :string)
  (h/html input))

(defn ensure-list [x]
  (if (or (nil? x) (seq? x))
    x
    (list x)))

(defmethod ig/init-key
  :mathdoc.process.write/htmlclean
  [_ v]
  (walk/postwalk
   (fn [c]
     (if (sequential? c)
       ((if (vector? c) vec identity)
        (mapcat ensure-list c))
       c))
   [:div.slides
    v
    [:script {:src "/js/main.js"}]]))
