(ns mathdoc.compiler.resources
  (:require [clojure.java.classpath :as cp]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [integrant.core :as ig]
            [mathdoc.utils.serialize :refer [serialize]]
            [duct.core :as duct]
            [duct.core :as duct]
            [mathdoc.core :as core])
  (:import java.io.File
           java.util.jar.JarFile))

(defn relative-file-seq [parent]
  (->>
   (file-seq (io/file parent))
   (remove #(.isDirectory ^File %))
   (map (partial core/relative-path parent))))

(defn cp-jarfiles-seq [prefix]
  (->>
   (cp/classpath)
   (filter cp/jar-file?)
   (filter #(str/starts-with? (.getName ^File %) prefix))
   (mapcat #(cp/filenames-in-jar (JarFile. ^File %)))))

(defn cp-directories-seq []
  (mapcat relative-file-seq (cp/classpath-directories)))

(defn copy-plan [paths files]
  (for [path paths]
    [path
     (->>
      files
      (keep
       (fn [file]
         (when (str/starts-with? file path)
           (str/replace-first
            file
            (re-pattern (str "^" (str/re-quote-replacement (str path "/"))))
            "")))))]))

(defn execute-plan [plan target logger]
  (doseq [[path files] plan]
    (doseq [file files]
      (duct/log logger :info :copy [path file target])
      (io/make-parents (io/file target file))
      (io/copy
       (serialize (io/resource (str path "/" file)))
       (io/file target file)))))

(derive ::copy :duct/compiler)

(defmethod ig/init-key
  ::copy
  [_ {:keys [prefix paths target logger]}]
  (let [files (concat (cp-directories-seq) (cp-jarfiles-seq prefix))]
    (execute-plan (copy-plan paths files)
                  target logger))
  target)

(comment

  ;; TODO: collect the paths in target into the deploy directory

  (ig/init-key
   ::copy {:prefix "mathdoc"
           :paths ["mathdoc/notes/public"
                   "mathdoc/cljs/public"
                   "mathdoc/server/public"]
           :target "target/mathdoc/resources/public"})

  (cp-directories-seq)

  (cp/classpath-jarfiles)

  (classpath-jarfiles-seq "common")

  (->> (cp/classpath-jarfiles)
       (take 2)
       (mapcat cp/filenames-in-jar)
       (take 100))

  (->>
   (cp/classpath-jarfiles)
   (map #(.getName %))
   (map #(io/file %))
   (map #(.getName %))
   (filter #(str/starts-with? "mathdoc" %)))

  (relative-file-seq "/home/david/p/mathdoc/mathdoc.dev/src")

  (mapcat relative-file-seq (cp/classpath-directories))

  )
