(ns mathdoc.core
  (:require [integrant.core :as ig]
            [clojure.spec :as s]
            [taoensso.tufte :as tufte]
            [clojure.spec :as s]
            [duct.core.protocols :as dcp]
            [duct.core :as duct]
            [clojure.java.io :as io]))

;;; specs

(s/def ::logger
  #(satisfies? dcp/Logger %))
(s/def ::root string?)
(s/def ::port int?)

;;; implementation

(defmethod ig/init-key
  ::port
  [_ port]
  {:pre [(s/assert ::port port)]}
  port)

(defmethod ig/init-key
  ::root
  [_ root]
  {:pre [(s/assert ::root root)]}
  root)

;;; config sources

(s/def ::sources (s/coll-of keyword?))

(defmethod ig/init-key
  ::config-sources
  [_ sources]
  {:pre [(s/assert ::sources sources)]}
  sources)

;; config

(defn read-config [& sources]
  (->> sources
       (flatten)
       (partition 2)
       (map
        (fn [[variant source]]
          (case variant
            :resource (io/resource source)
            :file (io/file source))))
       (apply duct/read-config)))

(defn exec
  [config]
  (let [system (-> config duct/prep ig/init)]
    (duct/add-shutdown-hook ::exec #(ig/halt! system))
    (.. Thread currentThread join)))

;; profiling

(defn profile-init-key [k v]
  (tufte/p k (#'ig/init-key k v)))

(defmethod ig/init-key
  ::profile?
  [_ v]
  (s/assert boolean? v))

(defn profile-init
  [{:keys [::profile?] :as config}]
  (ig/build config (keys config) profile-init-key))


;;; tests

(comment

  (read-config
   :resource "mathdoc/core/config.edn"
   :resource "mathdoc/server/config.edn"
   :resource "mathdoc/compiler/config.edn")

  )
