(ns mathdoc.server.watch
  (:require
   [clojure.java.io :as io]
   [clojure.spec :as s]
   [integrant.core :as ig]
   [clojure.string :as str]
   [hawk.core :as hawk]
   [clojure.core.async :as a]
   [duct.core :as duct])
  (:import
   java.net.URI
   java.io.File))

;;; specs

(s/def ::root string?)
(s/def ::content-dir string?)
(s/def ::process-fn fn?)

(s/def ::process
  (s/keys :req-un [::process-fn ::content-dir ::root]))

;;; implementation

(defn- get-relative-path
  "Path of `child` relative to `parent` as string. Returns `nil` if `child` is not a child of `parent`"
  [parent child]
  (let [child-uri ^URI (.toURI (io/file child))
        relative-child-uri ^URI (.relativize
                                 ^URI (.toURI (io/file parent))
                                 child-uri)]
    (when-not (= child-uri relative-child-uri)
      (.toString relative-child-uri))))

(defn- exists-directory? [^File dir]
  (and (.exists dir) (.isDirectory dir)))

(defn- ^File get-dir
  [{:keys [content-dir root]}]
  {:pre [(s/assert ::content-dir content-dir)
         (s/assert ::root root)]}
  (.getCanonicalFile (io/file root content-dir)))

(defn take-until-timeout [in]
  (let [time-out (a/timeout 100)]
    (a/go-loop [collect []]
      (when-let [[v ch] (a/alts! [in time-out])]
        (if (= ch time-out)
          collect
          (recur (conj collect v)))))))

(defn throttle-pipe [ch throttle-ch]
  (a/go-loop []
    (when-let [v (a/<! ch)]
      (let [vs (a/<! (take-until-timeout ch))]
        (when-let [vs (distinct (cons v vs))]
          (a/>! throttle-ch vs)))
      (recur))))

(defn consume-chan [ch f]
  (a/go-loop []
    (when-let [x (a/<! ch)]
      (f x)
      (recur))))

(defmethod ig/halt-key!
  ::process
  [_ {:keys [watcher raw-ch throttle-ch] :as this}]
  (some-> this :stop-watch (apply nil))
  (Thread/sleep 200)
  (some-> this :raw-ch a/close!)
  (some-> this :throttle-ch a/close!)
  (some-> this :loop-ch a/close!))

(defn- watch
  [paths ch]
  (partial hawk/stop!
     (hawk/watch!
      [{:paths   paths
        :filter  hawk/file?
        :handler (fn hawk-watch [_ ev]
                   (a/put! ch ev))}])))

(defmethod ig/init-key
  ::process
  [_ {:keys [logger process-fn] :as config}]
  {:pre [(s/assert ::process config)
         logger]}

  (duct/log logger :info :watch (str (get-dir config)))
  (assert (exists-directory? (get-dir config)) "watch dir does not exist")

  (let [raw-xf      (comp
                     (map (comp str :file))
                     (keep #(get-relative-path (get-dir config) %))
                     (filter #(str/ends-with? % ".md")))
        raw-ch      (a/chan (a/sliding-buffer 100) raw-xf)
        throttle-ch (a/chan (a/sliding-buffer 100))]
    (throttle-pipe raw-ch throttle-ch)
    {:raw-ch      raw-ch
     :throttle-ch throttle-ch
     :loop-ch     (consume-chan throttle-ch
                                (fn safe-process-fn [files]
                                  (try
                                    (process-fn files)
                                    (catch Throwable e
                                      (duct/log logger :error :process-error e)))) )
     :stop-watch  (watch [(get-dir config)] raw-ch)}))
