(ns mathdoc.endpoint.page
  (:require [clojure
             [spec :as s]
             [string :as str]]
            [clojure.java.io :as io]
            [compojure.core :as compojure]
            [duct.core :as duct]
            [integrant.core :as ig]
            [ring.util.response :as response]
            [selmer.parser :as selmer]
            [yaml.core :as yaml]
            [taoensso.timbre :as log]))

;; check that page exists
;; slurp html fragment
;; get metadata from files
;; generate meta data
;; determine correct template from meta data
;; add fragment as context for template
;; meta data also for templates (maybe separate file)

(s/def ::path
  (s/and string?
         not-empty
         #(not (str/ends-with? % "/"))))

(s/def ::fragment-path
  ::path)

(s/def ::content-path
  ::path)

(s/def ::template-path
  ::path)

(s/def ::config
  (s/keys
   :req-un [::content-path
            ::fragment-path
            ::template-path
            :mathdoc.core/logger]))


(def meta-regex #"(?m)(?s)^---\n(.*)^---")

(defn extract-metadata [s]
  (second (re-find meta-regex s)))

(defn maybe-slurp [f]
  (when (.exists (io/file f))
    (slurp f)))

#_ (maybe-slurp "project.clj")

(defn parse-yaml [s]
  (as-> s _
    (yaml/parse-string _)
    (if (vector? _) (first _) _)))

(defn path-metadata
  [path]
  (s/assert ::path path)
  #_(throw (ex-info "path-metadata" {:path path}))
  (some->
   (str path "/_metadata.yaml")
   (maybe-slurp)
   (parse-yaml)))

#_ (path-metadata "../talks/content")

(defn page-metadata
  [{:keys [content-path]} page]
  (some->
   page
   (as-> _ (str content-path "/" _ ".md"))
   (maybe-slurp)
   (extract-metadata)
   (parse-yaml)))

(defn page-yaml-metadata
  [{:keys [content-path]} page]
  (some->
   page
   (as-> _ (str content-path "/" _ ".yaml"))
   (maybe-slurp)
   (parse-yaml)))


(defn page-index-metadata
  [{:keys [content-path]} page]
  (some->
   page
   (as-> _ (str content-path "/" _ "/index.yaml"))
   (maybe-slurp)
   (parse-yaml)))

#_ (page-metadata {} nil)

(defn add-page-metadata
  [{:keys [content-path] :as config} page]
  (assoc-in
   config
   [:context :page]
   (merge
    (path-metadata
     (as-> page _
       (str content-path "/" _)
       (str/split _ #"/")
       (drop-last _)
       (str/join "/" _)))
    (page-index-metadata config page)
    (page-yaml-metadata config page)
    (page-metadata config page))))

(defn add-page-body
  [{:keys [fragment-path] :as config} page]
  (assoc-in
   config
   [:context :body]
   (maybe-slurp (io/file fragment-path (str page ".html")))))

(defn template-metadata
  [{:keys [template-path]} template]
  (some->
   template
   (str/replace #"\.html$" "")
   (as-> _ (str template-path "/" _ ".yaml"))
   (maybe-slurp)
   (parse-yaml)))

(defn get-page-template [config]
  (get-in config [:context :page :template]))

(defn add-template-metadata
  [{:keys [template-path context] :as config}]
  (assoc-in
   config
   [:context :template]
   (merge
    (path-metadata template-path)
    (template-metadata config (get-page-template config)))))

(defn indirect-include
  [args context-map]
  (selmer/render-file
   (get context-map (keyword (first args)))
   context-map))

(defn render
  [{:keys [template-path context] :as config}]
  (selmer/set-resource-path! (.toURL (io/file template-path)))
  (selmer/add-tag! :indirect-include indirect-include)
  #_(throw (ex-info "bad" config))
  (selmer/render-file (get-page-template config) context))

#_ (.exists (io/file "../talks/content" "interview/eth.md"))

(defn response
  [{:keys [content-path logger] :as config} req page]
  (duct/log logger :info :page [content-path page])
  (when (or (.exists (io/file content-path (str page ".yaml")))
            (.exists (io/file content-path (str page "/index.yaml")))
            (.exists (io/file content-path (str page ".md"))))
    (->
     config
     (assoc-in [:context :request] req)
     (add-page-metadata page)
     (add-page-body page)
     (add-template-metadata)
     (render)
     (response/response)
     (response/content-type "text/html; charset=utf-8"))))

(defmethod ig/init-key
  ::handler
  [_ {:keys [logger] :as config}]
  (s/assert ::config config)
  (compojure/routes
   (compojure/GET
    "/"
    req
    (duct/log logger :info :response "index")
    (response config req "index"))
   (compojure/GET
    "/:page/"
    [page :as req]
    (duct/log logger :info :response page)
    (response config req page))
   (compojure/GET
    "/:page/:subpage/"
    [page subpage :as req]
    (duct/log logger :info :response [page subpage])
    (response config req (str page "/" subpage)))))

(comment

  ((ig/init-key
    ::handler
    {:root "../talks"
     :target-dir "html"})
   (ring.mock.request/request :get "/test/"))

  (-> integrant.repl/system
      meta
      ::ig/build
      ::handler)

  (-> integrant.repl/system
      meta
      ::ig/build
      (get [:mathdoc.middleware/prone :mathdoc.core/var]))

  (-> integrant.repl/system
      (get [:mathdoc.middleware/prone :mathdoc.core/var]))


  (response {:template-path "../talks/template"
             :fragment-path "target/mathdoc/fragment/html"
             :content-path "../talks/content"}
            "eth")

  (page-body
   {:fragment-path "target/mathdoc/fragment/html"}
   "princeton")

  (page-body
   {:fragment-path "target/mathdoc/fragment/html"}
   "epfl")

  (page-metadata
   {:content-path "../talks/content"}
   "eth")

  (page-metadata*
   {:content-path "../talks/content"}
   "eth")

  (template-metadata {:template-path "../talks/template"} "layout/talk")

  )
