(ns mathdoc.process.transform.section
  (:require [clojure
             [spec :as s]
             [walk :as walk]]
            [taoensso.timbre :as log]
            [integrant.core :as ig]
            [mathdoc.specs.process :as msp]
            [duct.core :as duct]))

;;; process specs

(s/fdef process
        :args (s/cat :data (s/keys :req-un [::msp/ast]))
        :ret ::msp/ast)

;;; implementation


(defn wrap-div [attr elems]
  {:type :Div
   :content
   {:attr attr
    :elems elems}})

(defn section-div [{{{:keys [level attr]} :content :as header} :header
                    body :body}]
  (wrap-div
   (as-> attr _
     (update _ :id #(str "section-" %))
     (update _ :data-map assoc :section level)
     (update _ :class into ["section"
                            (str "level" level)]))
   (vec
    (cons header
          body))))



;;; specs

(defn header-spec [level]
  (s/and map?
         (comp #{:Header} :type)
         (comp #{level} :level :content)))

(defn upper-header-spec
  "matches headers of lower level (further up the hierarchy)"
  [level]
  (s/and map?
         (comp #{:Header} :type)
         (comp #(< % level) :level :content)))

(defn spec-complement [s]
  (comp not (partial s/valid? s)))

(defn not-header-spec [level]
  (spec-complement (header-spec level)))

(defn sections-spec [level]
  (s/and
   (spec-complement map?)
   (s/cat
    :upper (s/? (upper-header-spec level))
    :pre (s/* (not-header-spec level))
    :sections (s/+
               (s/cat
                :header (header-spec level)
                :body (s/* (not-header-spec level)))))))



;;; process

(defn process-sections [level v]
  (if-not (and (vector? v)
               (every? map? v)
               (some (comp #{:Header} :type) v))
    v
    (let [sspec (sections-spec level)
          conformv (s/conform sspec v)]
      (if (s/invalid? conformv)
        v
        (let [{:keys [upper pre sections]} conformv]
          (vec
           (cond->> (map section-div sections)
             pre (cons (wrap-div {:class [(str "presection-div-" level)]
                                  :data-map {:presection level}} pre))
             upper (cons upper))))))))

(defn walk-sections
  ([tree]
   (->> tree
        (walk-sections 1)
        (walk-sections 2)
        (walk-sections 3)
        (walk-sections 4)))
  ([level tree]
   (->>
    tree
    (walk/postwalk (partial process-sections level)))))

;; tree-seq

(defn process [{:keys [ast logger]}]
  (duct/log logger :info :process)
  (walk-sections ast))


;;; integrant

(defmethod ig/init-key
  :mathdoc.process.transform/section
  [_ v]
  (process v))



;;; tests


#_(-> reloaded.repl/system :section mathdoc.boundary.Data/get-data)


#_ (walk-sections
    #_2
    (concat (repeat 2 (header 3))
            [(header 2)]
            (repeat 2 (header 3))
            (repeat 2 (header 4))
            [(header 2)]
            (repeat 2 (header 3))
            (repeat 2 (header 4))
            [(header 2)]
            (repeat 2 (header 3))))

#_(walk-sections
   (concat (repeat 2 (header 3))
         [(header 2)]
         (repeat 2 (header 3))
         [(header 2)]
         (repeat 2 (header 3))
         [(header 2)]
         (repeat 2 (header 3))))

#_ (s/conform
    (section-spec 2)
    (concat (repeat 2 (header 3))
            [(header 2)]
            (repeat 2 (header 3))
            [(header 2)]
            (repeat 2 (header 3))
            [(header 2)]
            (repeat 2 (header 3))))


#_ (process-sections 2
    (concat (repeat 2 (header 3))
            [(header 2)]
            (repeat 2 (header 3))
            [(header 2)]
            (repeat 2 (header 3))
            [(header 2)]
            (repeat 2 (header 3))))

#_(-> reloaded.repl/system :section mathdoc.boundary.Data/get-data)

#_(-> reloaded.repl/system :section :data)

#_ (time (walk/postwalk #(process-sections (sections-spec 1) %) (-> reloaded.repl/system :section :input :data)))
#_ (time (walk/prewalk #(process-sections (sections-spec 1) %) (-> reloaded.repl/system :section :input :data)))
#_ (time (walk/prewalk #(process-sections 1 %) (-> reloaded.repl/system :section :input :data)))

#_ (time (walk-sections  (-> reloaded.repl/system :section :input :data)))
