(ns org.dthume.goad.executors
  (:require [clojure.core.async :refer [<!! >!! chan close! put! thread]])
  (:import [java.util.concurrent
            Callable ArrayBlockingQueue BlockingQueue
            ExecutorService Executors
            RejectedExecutionException RejectedExecutionHandler
            ThreadFactory ThreadPoolExecutor TimeUnit]))

(defprotocol AsyncExecutorService
  (shutdown-executor! [service timeout-ms])
  (submit-task [service f reply-ch close-reply?]))

(extend-protocol AsyncExecutorService
  ExecutorService
  (shutdown-executor! [this timeout-ms]
    (.shutdown this)
    (try
      (when (.awaitTermination this timeout-ms TimeUnit/MILLISECONDS)
        (.shutdownNow this))
      (catch InterruptedException e
        (.shutdownNow this)
        (.interrupt (Thread/currentThread)))))
  (submit-task
    [this f reply-ch close-reply?]
    (letfn [(close-reply [& args]
              (when close-reply? (close! reply-ch)))
            (task []
              (try
                (if-let [result (f)]
                  (put! reply-ch result close-reply)
                  (close-reply))
                (catch Throwable e
                  (put! reply-ch e close-reply))))]
      (let [^Callable callable (cast Callable task)]
        (.submit this callable)))))

(defn threadpool-executor-service
  ^ExecutorService [conf]
  (let [{:keys [^int core-pool-size
                ^int max-pool-size
                ^long keep-alive-ms
                ^ThreadFactory thread-factory
                ^RejectedExecutionHandler rejected-execution-handler
                ^BlockingQueue queue
                queue-size]
         :or {core-pool-size (int 0)
              keep-alive-ms Long/MAX_VALUE}} conf
        queue (or queue
                  (ArrayBlockingQueue.
                   (int (or queue-size max-pool-size))))
        ^ThreadPoolExecutor executor
        (ThreadPoolExecutor. core-pool-size
                             max-pool-size
                             keep-alive-ms
                             TimeUnit/MILLISECONDS
                             queue)]
    (when thread-factory
      (.setThreadFactory executor thread-factory))
    (when rejected-execution-handler
      (.setRejectedExecutionHandler executor rejected-execution-handler))
    executor))

(defn- as-executor-service
  [definition]
  (cond
   (instance? ExecutorService definition) definition
   (map? definition) (threadpool-executor-service definition)
   :else (throw (IllegalArgumentException.
                 (str "Cannot create ExecutorService from " definition)))))

(defn submit
  ([executor f]
     (submit f (chan)))
  ([executor f reply-ch]
     (submit f reply-ch true))
  ([executor f reply-ch close-reply?]
     (submit-task executor f reply-ch close-reply?)))

(defn- execute-worker-task
  [f args executor-service
   completion-watcher reply-ch close-reply-channel?]
  (try
    (let [future-result (submit-task executor-service
                                     #(apply f args)
                                     reply-ch close-reply-channel?)]
      (swap! completion-watcher (constantly future-result)))
    (catch RejectedExecutionException e
      (when-let [future-result @completion-watcher]
        (deref future-result 5000 ::not-done))
      (execute-worker-task f args executor-service
                           completion-watcher
                           reply-ch close-reply-channel?))))

(defn- worker-thread
  [f {:keys [close-reply-channel?
             completion-watcher
             executor-service
             input-channel
             shutdown-timeout-ms]
      :as opts}]
  (thread
    (loop [msg (<!! input-channel)]
      (if msg
        (let [[args reply-ch] msg
              result (apply f args)]
          (execute-worker-task f args executor-service completion-watcher
                               reply-ch close-reply-channel?)
          (recur (<!! input-channel)))
        ;; shutdown when input channel closes
        (try
          (shutdown-executor! executor-service shutdown-timeout-ms)
          (catch Throwable e e))))))

(defn- worker-fn
  [worker-t {:keys [input-channel] :as opts}]
  (with-meta
    (fn [& args]
      (let [reply-ch (chan)]
        (put! input-channel [args reply-ch])
        reply-ch))
    {::shutdown!
     (fn []
       (close! input-channel)
       worker-t)}))

(defn- worker-opts
  [opts]
  (-> {:input-channel (chan)
       :close-reply-channel? true
       :shutdown-executor? true
       :shutdown-timeout-ms 1000
       :completion-watcher (atom nil)}
      (into opts)
      (update-in [:executor-service] as-executor-service)))

(defn worker
  [f opts]
  (let [opts (worker-opts opts)
        worker-t (worker-thread f opts)]
    (worker-fn worker-t opts)))

(defn shutdown-worker!
  [w]
  (if-let [shutdown! (-> w meta ::shutdown!)]
    (shutdown!)
    (throw (IllegalArgumentException.
            (str "Cannot shutdown non-worker: " w)))))
