(ns erinite.user
  "Tools for interactive development with the REPL. This file should
  not be included in a production build of the application."
  (:require
    [environ.core :refer [env]]
    [erinite.figwheel :refer [start-figwheel stop-figwheel dump-figwheel-conf]]
    [com.stuartsierra.component :as component]
    [erinite.core.lib :as erinite-lib]
    [clojure.java.io :as io]
    [clojure.java.javadoc :refer [javadoc]]
    [clojure.pprint :refer [pprint]]
    [clojure.reflect :refer [reflect]]
    [clojure.repl :refer [apropos dir doc find-doc pst source]]
    [clojure.set :as set]
    [clojure.string :as str]
    [clojure.test :as test]
    [clojure.tools.namespace.repl :refer [refresh refresh-all] :as repl]))

;(repl/set-refresh-dirs "src/")

(defn help
  []
  (println "Erinite Help")
  (println "  (help)          Erinite help")
  (println "  (build :start)  Start or restart figwheel autobuild")
  (println "  (build :stop)   Stop figwheel autobuild")
  (println "  (build :dump)   Display the configuration used")
  (println "  (reset)         Reload and restart your erinite system")
  (println "  system          Var containing application system")
  (println))


(defn build
  "Control figwheel."
  [action]
  (case action
    :start  (start-figwheel)
    :stop   (stop-figwheel)
    :dump   (dump-figwheel-conf)))


(def system
  "A Var containing an object representing the application under
  development."
  nil)


(defn init
  "Creates and initializes the system under development in the Var
  #'system."
  []
  (let [system-sym (env :-system)]
    (println "Creating system using" system-sym)
    (-> system-sym symbol namespace symbol require)
    (let [system-ctor (resolve system-sym)]
      (alter-var-root
        #'system
        (constantly (erinite-lib/get-system system-ctor))))))


(defn start
  "Starts the system running, updates the Var #'system."
  []
  (println "Starting API Server")
  (alter-var-root #'system component/start))


(defn stop
  "Stops the system if it is currently running, updates the Var
  #'system."
  []
  (println "Stopping API Server")
  (alter-var-root #'system
    (fn [s] (when s (component/stop s)))))


(defn go
  "Initializes and starts the system running."
  []
  (init)
  (start)
  :ready)


(defn reset
  "Stops the system, reloads modified source files, and restarts it."
  []
  (stop)
  (refresh :after 'erinite.user/go))

(defn h->m
  "In a clj-http response map, convert HeaderMap to an ordinary map.
   NOTE: This is a temp workaround to ultra/whidbey not being able to pprint
    HeaderMaps and is useful for doing test http requests.
   This will likely be removed and can be ignored by users who either don't use
   ultra or whidbey, or users who don't use clj-http from the REPL"
  [response-map]
  (update-in response-map [:headers] (partial into {})))

