(ns org.euandre.http.components
  (:require [com.stuartsierra.component :as component]
            clojure.java.io
            [clojure.edn :as edn]
            [datomic.api :as d]
            [org.euandre.misc.core :as misc]
            [io.pedestal.http :as http]
            [io.pedestal.interceptor :as i]))

(defn- test? [{:keys [env]}]
  (not (#{:staging :prod :dev} env)))

(defprotocol IWebAppComponent
  (webapp-component [this] "Fetch the relevant part to be usued as a WebApp component."))

(defrecord Config [config]
  component/Lifecycle
  (start [this]
    (let [edn-config (-> "config.edn" clojure.java.io/resource slurp edn/read-string)]
      (assoc this :config edn-config)))
  (stop [this]
    (assoc this :config nil))
  IWebAppComponent
  (webapp-component [this]
    (:config this)))

(defn new-config []
  (map->Config {}))

(defrecord Datomic [config conn schemas]
  component/Lifecycle
  (start [this]
    (let [uri           (get-in config [:config :datomic-uri])
          new-database? (d/create-database uri)
          conn          (d/connect uri)]
      (when new-database?
        @(d/transact conn schemas))
      (assoc this :conn conn)))
  (stop [this]
    (when conn (d/release conn))
    (assoc this :conn nil))
  IWebAppComponent
  (webapp-component [this]
    (:conn this)))

(defn new-datomic [schemas]
  (map->Datomic {:schemas schemas}))

(defn adapt-to-webapp-components [webapp]
  (misc/map-vals (fn [component]
                   (if (satisfies? IWebAppComponent component)
                     (webapp-component component)
                     component))
                 webapp))

(defn inject-components [webapp]
  (i/interceptor
   {:name  ::inject-components
    :enter #(assoc-in % [:request :components] (adapt-to-webapp-components webapp))}))

(defn- my-system-global-interceptors [service-map webapp]
  (update service-map
          ::http/interceptors
          (fn [interceptors-vector]
            (vec (cons (inject-components webapp) interceptors-vector)))))

(defrecord Pedestal [service-map datomic service config]
  component/Lifecycle
  (start [this]
    (if service
      this
      (cond-> service-map
        true                      (assoc ::http/port (get-in config [:config :port]))
        true                      http/default-interceptors
        (test? service-map)       http/dev-interceptors
        true                      (my-system-global-interceptors this)
        true                      http/create-server
        (not (test? service-map)) http/start
        true                      (#(assoc this :service %)))))
  (stop [this]
    (when (and service
               (not (test? service-map)))
      (http/stop service))
    (assoc this :service nil)))

(defn new-pedestal []
  (map->Pedestal {}))
