(ns org.euandre.http.components
  (:require [buddy.auth.backends :as auth.backends]
            [com.stuartsierra.component :as component]
            clojure.java.io
            [clojure.edn :as edn]
            [datomic.api :as d]
            [org.euandre.http.protocols.edn :as protocols.edn]
            [org.euandre.misc.core :as misc]
            [io.pedestal.http :as http]
            [io.pedestal.interceptor :as i]
            [clojure.java.io :as io]))

(defprotocol IWebAppComponent
  (webapp-component [this] "Fetch the relevant part to be usued as a WebApp component."))

(defrecord Config [config-file-name config]
  component/Lifecycle
  (start [this]
    (let [edn-config (-> config-file-name clojure.java.io/resource slurp edn/read-string)]
      (assoc this :config edn-config)))
  (stop [this]
    (assoc this :config nil))
  IWebAppComponent
  (webapp-component [this]
    (:config this)))

(defn new-config [config-file-name]
  (map->Config {:config-file-name config-file-name}))

(defn epoch-seconds->date [n]
  (java.util.Date. (* 1000 n)))

(defn jws-backend [public-key]
  (auth.backends/jws {:secret     public-key
                      :token-name "Bearer"
                      :options    {:alg :rs256}
                      :authfn     (fn [{:keys [exp iat scopes jti sub] :as claims}]
                                    (merge claims
                                           {:expires-at (epoch-seconds->date exp)
                                            :issued-at  (epoch-seconds->date iat)}))}))

(defrecord Crypto [config jwt-backend]
  component/Lifecycle
  (start [this]
    (let [public-key (-> config
                         (get-in [:config :auth-certificate-file])
                         io/resource
                         buddy.core.keys/public-key)]
      (assoc this :jws-backend (jws-backend public-key))))
  (stop [this]
    (assoc this :jws-backend nil))
  IWebAppComponent
  (webapp-component [this]
    (:jws-backend this)))

(defn new-crypto []
  (map->Crypto {}))

(defrecord Datomic [config conn schemas]
  component/Lifecycle
  (start [this]
    (let [uri           (get-in config [:config :datomic-uri])
          new-database? (d/create-database uri)
          conn          (d/connect uri)]
      (when new-database?
        @(d/transact conn schemas))
      (assoc this :conn conn)))
  (stop [this]
    (when conn (d/release conn))
    (assoc this :conn nil))
  IWebAppComponent
  (webapp-component [this]
    (:conn this)))

(defn new-datomic [schemas]
  (map->Datomic {:schemas schemas}))

(defn adapt-to-webapp-components [webapp]
  (misc/map-vals (fn [component]
                   (if (satisfies? IWebAppComponent component)
                     (webapp-component component)
                     component))
                 webapp))

(defn inject-components [webapp]
  (i/interceptor
   {:name  ::inject-components
    :enter #(assoc-in % [:request :components] (adapt-to-webapp-components webapp))}))

(defn- my-system-global-interceptors [service-map webapp]
  (update service-map
          ::http/interceptors
          (fn [interceptors-vector]
            (vec (cons (inject-components webapp) interceptors-vector)))))

(defn- test? [{:keys [env]}] (#{:test} env))
(defn- dev? [{:keys [env]}]  (#{:test :dev} env))

(defrecord Pedestal [service-map datomic service config]
  component/Lifecycle
  (start [this]
    (if service
      this
      (cond-> service-map
        true                      (assoc ::http/port (get-in config [:config :port]))
        true                      http/default-interceptors
        (dev? service-map)        http/dev-interceptors
        true                      (my-system-global-interceptors this)
        true                      http/create-server
        (not (test? service-map)) http/start
        true                      (#(assoc this :service %)))))
  (stop [this]
    (when (and service
               (not (test? service-map)))
      (http/stop service))
    (assoc this :service nil)))

(defn new-pedestal []
  (map->Pedestal {}))

(defrecord EdnAtomFileImpl [config-key config]
  protocols.edn/EdnAtomFile
  (read! [this]
    (-> (file this) slurp clojure.edn/read-string))
  (write! [this new-database]
    (swap! (:database-atom this) (constantly new-database))
    (spit (file this) new-database))
  (file [_]
    (some-> config :config config-key clojure.java.io/resource))
  (db [this]
    @(:database-atom this))

  component/Lifecycle
  (start [this]
    (let [database-atom (atom (read! this))]
      (assoc this :database-atom database-atom)))
  (stop [this]
    (write! this @(:database-atom this))
    (assoc this :database-atom nil))
  http.components/IWebAppComponent
  (webapp-component [this]
    (:database-atom this)))

(defn new-edn-database [config-key]
  (map->EdnAtomFileImpl {:config-key config-key}))
