(ns org.euandre.http.components
  (:require [buddy.auth.backends :as auth.backends]
            [com.stuartsierra.component :as component]
            clojure.java.io
            [clojure.edn :as edn]
            [org.euandre.http.protocols.edn :as protocols.edn]
            [org.euandre.misc.core :as misc]
            [overtone.at-at :as at-at]
            [io.pedestal.http :as http]
            [io.pedestal.interceptor :as i]
            [clojure.java.io :as io]))

(defprotocol IWebAppComponent
  (webapp-component [this] "Fetch the relevant part to be usued as a WebApp component."))

(defrecord Config [config-file-name config]
  component/Lifecycle
  (start [this]
    (let [edn-config (-> config-file-name clojure.java.io/resource slurp edn/read-string)]
      (assoc this :config edn-config)))
  (stop [this]
    (assoc this :config nil))
  IWebAppComponent
  (webapp-component [this]
    (:config this)))

(defn new-config [config-file-name]
  (map->Config {:config-file-name config-file-name}))

(defn epoch-seconds->date [n]
  (java.util.Date. (* 1000 n)))

(defn jws-backend [public-key]
  (auth.backends/jws {:secret     public-key
                      :token-name "Bearer"
                      :options    {:alg :rs256}
                      :authfn     (fn [{:keys [exp iat scopes jti sub] :as claims}]
                                    (merge claims
                                           {:expires-at (epoch-seconds->date exp)
                                            :issued-at  (epoch-seconds->date iat)}))}))

(defrecord Crypto [config jwt-backend]
  component/Lifecycle
  (start [this]
    (let [public-key (-> config
                         (get-in [:config :auth-certificate-file])
                         io/resource
                         buddy.core.keys/public-key)]
      (assoc this :jws-backend (jws-backend public-key))))
  (stop [this]
    (assoc this :jws-backend nil))
  IWebAppComponent
  (webapp-component [this]
    (:jws-backend this)))

(defn new-crypto []
  (map->Crypto {}))

(defn adapt-to-webapp-components [webapp]
  (misc/map-vals (fn [component]
                   (if (satisfies? IWebAppComponent component)
                     (webapp-component component)
                     component))
                 webapp))

(defn inject-components [webapp]
  (i/interceptor
   {:name  ::inject-components
    :enter #(assoc-in % [:request :components] (adapt-to-webapp-components webapp))}))

(defn- my-system-global-interceptors [service-map webapp]
  (update service-map
          ::http/interceptors
          (fn [interceptors-vector]
            (vec (cons (inject-components webapp) interceptors-vector)))))

(defn- test? [{:keys [env]}] (#{:test} env))
(defn- dev? [{:keys [env]}]  (#{:test :dev} env))

(defrecord Pedestal [routes service-map service config]
  component/Lifecycle
  (start [this]
    (if service
      this
      (let [config-map (merge {::http/port   (get-in config [:config :port] 1234)
                               ::http/routes routes
                               ::http/join?  false ;; do not block thread that starts web (server)
                               ::http/type   :jetty}
                              service-map)]
        (cond-> config-map
          true                     http/default-interceptors
          (dev? config-map)        http/dev-interceptors
          (dev? config-map)        (assoc ::http/allowed-origins {:allowed-origins (constantly true)})
          true                     (my-system-global-interceptors this)
          true                     http/create-server
          (not (test? config-map)) http/start
          true                     (#(assoc this :service %))))))
  (stop [this]
    (when (and service
               (not (test? service-map)))
      (http/stop service))
    (assoc this :service nil)))

(defn new-pedestal [routes]
  (map->Pedestal {:routes routes}))

(defrecord EdnAtomFileImpl [config-path config]
  protocols.edn/EdnAtomFile
  (read! [this]
    (-> (protocols.edn/file this) slurp clojure.edn/read-string))
  (write! [this new-database]
    (swap! (:database-atom this) (constantly new-database))
    (spit (protocols.edn/file this) new-database))
  (file [_]
    (some-> config :config (get-in config-path)))
  (db [this]
    @(:database-atom this))

  component/Lifecycle
  (start [this]
    (let [database-atom (atom (protocols.edn/read! this))]
      (assoc this :database-atom database-atom)))
  (stop [this]
    (protocols.edn/write! this @(:database-atom this))
    (assoc this :database-atom nil)))

(defn new-edn-database [config-path]
  (map->EdnAtomFileImpl {:config-path config-path}))

(def default-pooling-ms 1000)

(defrecord BackgroundJob [job-fn config-path deps config database]
  component/Lifecycle
  (start [this]
    (let [pool    (at-at/mk-pool :cpu-count 1)
          wait-ms (get-in (:config config) config-path default-pooling-ms)]
      (at-at/every wait-ms (partial job-fn (select-keys this deps)) pool)
      (assoc this :pool pool)))
  (stop [this]
    (at-at/stop-and-reset-pool! (:pool this))
    (assoc this :pool nil))
  IWebAppComponent
  (webapp-component [this]
    (:pool this)))

(defn new-background-job [job-fn config-path deps]
  (map->BackgroundJob {:job-fn      job-fn
                       :config-path config-path
                       :deps        deps}))
