(ns org.euandre.http.components
  (:require [buddy.auth.backends :as auth.backends]
            [com.stuartsierra.component :as component]
            clojure.java.io
            [clojure.edn :as edn]
            [org.euandre.http.protocols.edn :as protocols.edn]
            [org.euandre.misc.core :as misc]
            [tea-time.core :as tea-time]
            [io.pedestal.http :as http]
            [io.pedestal.interceptor :as i]
            [clojure.java.io :as io]))

(defprotocol IWebAppComponent
  (webapp-component [this] "Fetch the relevant part to be used as a WebApp component."))

(def default-ttl (* 1000 60 5))
(defrecord EdnFile [config-file-path config-ttl-path config edn-file-fn]
  component/Lifecycle
  (start [this]
    (let [edn-file-fn #(-> config (get-in config-file-path) slurp edn/read-string)
          ttl         (get-in config config-ttl-path default-ttl)]
      (assoc this :edn-file-fn (clojure.core.memoize/ttl edn-file-fn :ttl/threshold ttl))))
  (stop [this]
    (assoc this :edn-file-fn nil))
  IWebAppComponent
  (webapp-component [this]
    (:edn-file-fn this)))

(defn new-edn-file [config-file-path config-ttl-path]
  (map->EdnFile {:config-file-path config-file-path
                 :config-ttl-path  config-ttl-path}))

(defn epoch-seconds->date [n]
  (java.util.Date. (* 1000 n)))

(defn jws-backend [public-key]
  (auth.backends/jws {:secret     public-key
                      :token-name "Bearer"
                      :options    {:alg :rs256}
                      :authfn     (fn [{:keys [exp iat scopes jti sub] :as claims}]
                                    (merge claims
                                           {:expires-at (epoch-seconds->date exp)
                                            :issued-at  (epoch-seconds->date iat)}))}))

(defrecord Crypto [config jwt-backend]
  component/Lifecycle
  (start [this]
    (let [public-key (-> config
                         (get-in [:auth-certificate-file])
                         io/resource
                         buddy.core.keys/public-key)]
      (assoc this :jws-backend (jws-backend public-key))))
  (stop [this]
    (assoc this :jws-backend nil))
  IWebAppComponent
  (webapp-component [this]
    (:jws-backend this)))

(defn new-crypto []
  (map->Crypto {}))

(defn adapt-to-webapp-components [webapp]
  (misc/map-vals (fn [component]
                   (if (satisfies? IWebAppComponent component)
                     (webapp-component component)
                     component))
                 webapp))

(defn inject-components [webapp]
  (i/interceptor
   {:name  ::inject-components
    :enter #(assoc-in % [:request :components] (adapt-to-webapp-components webapp))}))

(defn- my-system-global-interceptors [service-map webapp]
  (update service-map
          ::http/interceptors
          (fn [interceptors-vector]
            (vec (cons (inject-components webapp) interceptors-vector)))))

(defn- test? [{:keys [env]}] (#{:test} env))
(defn- dev? [{:keys [env]}]  (#{:test :dev} env))

(defrecord Pedestal [routes service-map service config]
  component/Lifecycle
  (start [this]
    (if service
      this
      (let [config-map (merge {::http/port   (get config :port 1234)
                               ::http/routes routes
                               ::http/join?  false ;; do not block thread that starts web (server)
                               ::http/type   :jetty}
                              service-map)]
        (cond-> config-map
          true                     http/default-interceptors
          (dev? config-map)        http/dev-interceptors
          (dev? config-map)        (assoc ::http/allowed-origins {:allowed-origins (constantly true)})
          true                     (my-system-global-interceptors this)
          true                     http/create-server
          (not (test? config-map)) http/start
          true                     (#(assoc this :service %))))))
  (stop [this]
    (when (and service
               (not (test? service-map)))
      (http/stop service))
    (assoc this :service nil)))

(defn new-pedestal [routes]
  (map->Pedestal {:routes routes}))

(defn slurp-if [f]
  (when (.exists (clojure.java.io/as-file f))
    (slurp f)))

(defrecord EdnAtomFileImpl [config-path config]
  protocols.edn/EdnAtomFile
  (read! [this]
    (or (some-> (protocols.edn/file this) slurp-if clojure.edn/read-string)
        {}))
  (write! [this new-database]
    (swap! (:database-atom this) (constantly new-database))
    (spit (protocols.edn/file this) new-database))
  (file [_]
    (get-in config config-path))
  (db [this]
    @(:database-atom this))

  component/Lifecycle
  (start [this]
    (let [database-atom (atom (protocols.edn/read! this))]
      (assoc this :database-atom database-atom)))
  (stop [this]
    (protocols.edn/write! this @(:database-atom this))
    (assoc this :database-atom nil)))

(defn new-edn-database [config-path]
  (map->EdnAtomFileImpl {:config-path config-path}))


(def default-pooling-sec 3)
(defrecord BackgroundJob [job-fn config-path deps config database]
  component/Lifecycle
  (start [this]
    (let [wait-sec (get-in config config-path default-pooling-sec)
          every    (tea-time/every! wait-sec (partial job-fn (select-keys this deps)))]
      (assoc this :tea-time-every every)))
  (stop [this]
    (tea-time/cancel! (:tea-time-every this))
    (assoc this :tea-time-every nil))
  IWebAppComponent
  (webapp-component [this]
    (:tea-time-every this)))

(defn new-background-job [job-fn config-path deps]
  (map->BackgroundJob {:job-fn      job-fn
                       :config-path config-path
                       :deps        deps}))
