(ns org.euandre.misc.edn.email
  "edn readers and custom printers for Email."
  (:require [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as s.gen]
            [miner.strgen :as strgen]))

;; Taken from:
;; https://clojure.org/guides/spec#_entity_maps
(def email-regex #"^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,63}$")

(defrecord Email [email-string]
  Object
  (toString [_] email-string))

(defn validate-email [email]
  (cond
    (and (not (string? email))
         (not (instance? Email email)))
    {::valid?             false
     ::reason             ::not-a-string
     ::reason-explanation (str "Input email '" email "' is not a string or #email.")}

    (not (re-matches email-regex (str email)))
    {::valid?             false
     ::reason             ::unrecognized-string
     ::reason-explanation (str "Input email string '" email "' is not recognized as a valid email.")}

    true
    {::valid?             true
     ::reason             ::valid
     ::reason-explanation "Valid email"}))

(defn string->email [email-string]
  (let [{::keys [valid? reason-explanation]} (validate-email email-string)]
    (assert valid? reason-explanation)
    (->Email email-string)))

(defmethod print-method Email [email ^java.io.Writer writer]
  (.write writer "#email ")
  (print-method (str email) writer))

(defmethod print-dup Email [email ^java.io.Writer writer]
  (.write writer "#email ")
  (print-method (str email) writer))

;; Specs

(s/def ::email-string
  (s/with-gen (s/and string? #(re-matches email-regex %))
    #(strgen/string-generator email-regex)))

(s/def ::email
  (s/with-gen #(instance? Email %)
    #(s.gen/fmap string->email
                 (s/gen ::email-string))))

(s/def ::valid? boolean?)
(s/def ::reason #{::not-a-string ::unrecognized-string ::valid})
(s/def ::reason-explanation string?)
(s/def ::validation-map
  (s/keys
   :req [::valid? ::reason ::reason-explanation]))

(s/fdef validate-email
  :args (s/cat :email (s/or :email-string ::email-string
                            :email-instance ::email
                            :others any?))
  :ret  ::validation-map)

(s/fdef string->email
  :args (s/cat :email-string ::email-string)
  :ret  ::email)
