(ns org.euandre.misc.spec.generators
  "Custom generators for spec definitions and test.check tests."
  (:require [clojure.spec.gen.alpha :as s.gen]
            [clojure.spec.alpha :as s]))

;; Derived from "Building test check Generators - Gary Fredericks", around minute 32
;; https://www.youtube.com/watch?v=F4VZPxLZUdA

(s/def ::year
  (s/with-gen int?
    #(s.gen/fmap (partial + 2017) (s.gen/int))))

(s/def ::month
  (s/with-gen (s/and int? #(<= 1 % 12))
    #(s.gen/large-integer* {:min 1 :max 12})))

(s/def ::day
  (s/with-gen (s/and int? #(<= 1 % 12))
    #(s.gen/large-integer* {:min 1 :max 31})))

(s/def ::hour
  (s/with-gen (s/and int? #(<= 0 % 12))
    #(s.gen/large-integer* {:min 0 :max 23})))

(s/def ::minute
  (s/with-gen (s/and int? #(<= 0 % 12))
    #(s.gen/large-integer* {:min 0 :max 59})))

(s/def ::second
  (s/with-gen (s/and int? #(<= 0 % 12))
    #(s.gen/large-integer* {:min 0 :max 59})))

(s/def ::millis
  (s/with-gen (s/and int? #(<= 0 % 12))
    #(s.gen/large-integer* {:min 0 :max 999})))

(s/def ::instant-map
  (s/keys
   :req [::year ::month ::day ::hour ::minute ::second ::millis]))

(defn- construct-instant
  "Not to be used for date parsing, just a helper function for generating useful instants."
  [instant-map]
  (letfn [(try-construct-instant
            [{::keys [year month day hour minute second millis] :as instant-map-inner}
             try-n]
            (when (> try-n 4)
              (throw (ex-info "Max tries (4) to build an Instant, invalid :day entry."
                              {:try-n        try-n
                               :original-day (+ (dec try-n) day)
                               :instant-map  instant-map-inner})))
            (let [format-str "%04d-%02d-%02dT%02d:%02d:%02d.%03dZ"]
              (try
                (java.time.Instant/parse
                 (format format-str
                         year month day hour minute second millis))
                (catch java.time.format.DateTimeParseException e
                  (try-construct-instant (update instant-map-inner ::day dec) (inc try-n))))))]
    (try-construct-instant instant-map 1)))

(s/fdef construct-instant
  :args (s/cat :instant-map ::instant-map)
  :ret #(instance? % java.time.Instant))

(def instant
  "Custom `java.time.Instant` generator, compatible with `clojure.test.check`."
  (s.gen/fmap construct-instant (s/gen ::instant-map)))
