(ns org.euandreh.http.test-aux
  (:require [io.pedestal.http.route :as route]
            [io.pedestal.http :as http]
            io.pedestal.test
            [clojure.edn :as edn]
            [com.stuartsierra.component :as component]))

(def ^:dynamic *headers*
  {"Content-Type" "application/edn"})

(defn url-for [routes route-identifier]
  (if (string? route-identifier)
    route-identifier
    ((route/url-for-routes routes) route-identifier)))

(defn request!
  ([method raw-response? service route-name status-code]
   (request! method raw-response? service route-name status-code nil))
  ([method raw-response? service route-name status-code request-body & options]
   (let [route-url (url-for (::http/routes service) route-name)
         response  (apply io.pedestal.test/response-for
                          (::http/service-fn service)
                          method
                          route-url
                          :headers *headers*
                          :body (pr-str request-body)
                          options)]
     (assert (= status-code (:status response))
             (do
               (str "Expected response status code to be " status-code ", got " (pr-str (:status response)))
               (pr-str (:body response))))
     (if raw-response?
       response
       (edn/read-string {:readers (merge *data-readers*
                                         {'error identity})}
                        (:body response))))))

(def GET-raw  (partial request! :get  true))
(def POST-raw (partial request! :post true))
(def GET      (partial request! :get  false))
(def POST     (partial request! :post false))

(defn query! [service status-code query]
  (POST service :query status-code query)) ;; FIXME: :query vs :graph

(defmacro with-system
  [system-fn system-under-test-var components-bindings & body]
  `(let [~system-under-test-var       (component/start (~system-fn))
         service#                     (get-in ~system-under-test-var [:pedestal :service]) ;; fixme
         GET#                         (partial GET service#)
         GET-raw#                     (partial GET-raw service#)
         POST#                        (partial POST service#)
         POST-raw#                    (partial POST-raw service#)
         query-raw#                   (partial query! service#)
         q#                           #(query-raw# 200 {:q %})
         {:keys ~components-bindings} {:query    query-raw#
                                       :q        q#
                                       :service  service#
                                       :GET      GET#
                                       :GET-raw  GET-raw#
                                       :POST     POST#
                                       :POST-raw POST-raw#}]
     (try
       ~@body
       (finally
         (component/stop ~system-under-test-var)))))

(defmacro with-headers [headers-map & body]
  `(binding [*headers* (merge org.euandreh.http.test-aux/*headers* ~headers-map)]
     ~@body))

(defmacro with-token [token & body]
  `(with-headers {"Authorization" ~token}
     ~@body))

(defmacro without-logs [& body]
  `(with-redefs [io.pedestal.log/-level-enabled? (constantly false)]
     ~@body))
