(ns org.euandreh.http.test-aux
  (:require [io.pedestal.http.route :as route]
            [io.pedestal.http :as http]
            io.pedestal.test
            [clojure.edn :as edn]
            [com.stuartsierra.component :as component]))

(def ^:dynamic *headers*
  {"Content-Type" "application/edn"})

(defn url-for [routes route-identifier]
  (if (string? route-identifier)
    route-identifier
    ((route/url-for-routes routes) route-identifier)))

(defn request!
  ([method raw-response? service route-name status-code]
   (request! method raw-response? service route-name status-code nil))
  ([method raw-response? service route-name status-code request-body & options]
   (let [route-url (url-for (::http/routes service) route-name)
         response  (apply io.pedestal.test/response-for
                          (::http/service-fn service)
                          method
                          route-url
                          :headers *headers*
                          :body (pr-str request-body)
                          options)]
     (assert (= status-code (:status response))
             {:error-msg (str "Expected response status code to be " status-code ", got " (pr-str (:status response))
                          )
              :body response})
     (if raw-response?
       response
       (edn/read-string {:readers (merge *data-readers*
                                         {'error identity})}
                        (:body response))))))

(def ^{:arglists '([service route-name status-code] [service route-name status-code request-body & options])}
  GET-raw "Perform an HTTP GET request to the given `service`. Return the body as a raw string."
  (partial request! :get true))
(def ^{:arglists '([service route-name status-code] [service route-name status-code request-body & options])}
  POST-raw "Perform an HTTP POST request to the given `service`. Return the body as a raw string."
  (partial request! :post true))
(def ^{:arglists '([service route-name status-code] [service route-name status-code request-body & options])}
  GET "Perform an HTTP GET request to the given `service`. Parse the return body as edn data."
  (partial request! :get false))
(def ^{:arglists '([service route-name status-code] [service route-name status-code request-body & options])}
  POST "Perform an HTTP POST request to the given `service`. Parse the return body as edn data."
  (partial request! :post false))

(defn query! [service status-code query]
  (POST service :graph status-code query))

(defmacro with-system
  [system-fn system-under-test-var components-bindings & body]
  `(let [~system-under-test-var       (component/start (~system-fn))
         service#                     (get-in ~system-under-test-var [:pedestal :service]) ;; fixme
         GET#                         (partial GET service#)
         GET-raw#                     (partial GET-raw service#)
         POST#                        (partial POST service#)
         POST-raw#                    (partial POST-raw service#)
         query-raw#                   (partial query! service#)
         q#                           #(query-raw# 200 {:q %})
         {:keys ~components-bindings} {:query    query-raw#
                                       :q        q#
                                       :service  service#
                                       :GET      GET#
                                       :GET-raw  GET-raw#
                                       :POST     POST#
                                       :POST-raw POST-raw#}]
     (try
       ~@body
       (finally
         (component/stop ~system-under-test-var)))))

(defmacro make-custom-with-system
  "Build a custom `with-system` macro to supress the need of always saying the system building function."
  [system-fn-sym macro-name]
  `(defmacro ~macro-name
     "Macro \"partial application\"."
     [system-under-test-var# components-bindings# & body#]
     `(~'org.euandreh.http.test-aux/with-system ~'~system-fn-sym ~system-under-test-var# ~components-bindings# ~@body#)))

(defmacro with-headers
  "Bind `*headers*` var to the custom received header map. Usually used in conjunction with `with-system`.

  (with-headers {\"X-Header\" \"header value\"}
    (with-system ...))"
  [headers-map & body]
  `(binding [*headers* (merge org.euandreh.http.test-aux/*headers* ~headers-map)]
     ~@body))

(defmacro with-token
  "Adds a custom token to the authorization header.

  (with-token \"Bearer xxxxx\"
    (with-system ...))"
  [token & body]
  `(with-headers {"Authorization" ~token}
     ~@body))

(defmacro without-logs
  "Disable pedestal logs inside the body. Useful for suppressing expected error messages when testing for specific cases of exceptions."
  [& body]
  `(with-redefs [io.pedestal.log/-level-enabled? (constantly false)]
     ~@body))
