(ns org.euandreh.http.exception
  "HTTP-related exceptions.

   When used in conjunction with the `org.euandreh.http.interceptors/catch` interceptor, these exceptions can be used to stop the control flow of the handling code and return a proper error message and status code to the client.

   `(exception/forbidden! {...})` throws an ExceptionInfo with metadata attached to it that `org.euandreh.http.interceptors/catch` knows how to find, and return a 403 status code with the `Forbidden` message."
  (:require [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as s.gen]))

(defn- exception
  "Build an `clojure.lang.ExceptionInfo` and return it.

   Embed the `status-code` and the `type` as values of the error map. These values are inspected upper on the stack to decide what type of HTTP response to send back to the client."
  [status-code message type error-map]
  (ex-info message
           (merge error-map
                  {:exception/status-code status-code
                   :exception/type        type})))

(defn- exception!
  "Build an exception with `exception` fn and actually throw it."
  [& exception-args]
  (throw (apply exception exception-args)))

(defn- expand-exception-fn
  "Expand the single triple entry into a function definition form.

  (expand-exception-fn [123 \"message\" :type])
  => (do
       (def my-type  (partial exception  123 \"message\" :my-type))
       (def my-type! (partial exception! 123 \"message\" :my-type)))"
  [[code message ex-name]]
  (let [fn-name (-> ex-name
                    name
                    symbol
                    (vary-meta assoc
                               :arglists ''([error-map])
                               :doc "Build and return the exception, but doesn't throw it."))
        fn-name! (-> ex-name
                     name
                     (str "!")
                     symbol
                     (vary-meta assoc
                                :arglists ''([error-map])
                                :doc "Build and throw the exception."))]
    `(do
       (def ~fn-name
         (partial exception ~code ~message ~ex-name))
       (def ~fn-name!
         (partial exception! ~code ~message ~ex-name)))))

(defmacro ^:private gen-exception-fns
  "Build all the exception throwing functions from the input data."
  [exception-input-data]
  `(do ~@(mapv expand-exception-fn exception-input-data)))

(gen-exception-fns
 [[400 "Invalid Input" :invalid-input]
  [401 "Unauthorized" :unauthorized]
  [403 "Forbidden" :forbidden]
  [404 "Not Found" :not-found]
  [409 "Conflict" :conflict]
  [412 "Precondition Failed" :precondition-failed]
  [429 "Too Many Requests" :too-many-requests]
  [500 "Server Error" :server-error]])

;; Specs

(s/def ::status-code
  (s/with-gen (s/and int? #(<= 400 % 599))
    #(s.gen/large-integer* {:min 400 :max 599})))

(s/def ::expand-exception-fn-args
  (s/tuple ::status-code string? keyword?))

(s/fdef expand-exception-fn
  :args (s/cat :input-tuple ::expand-exception-fn-args)
  :ret (fn [expansion]
         (let [defs (rest expansion)]
           (and (seq? expansion)
                (= 3 (count expansion))
                (= '[def def] (map first defs))
                (every? seq? (map #(nth % 2) defs))))))

(s/def ::exception-input
  (s/cat :status-code ::status-code
         :message string?
         :type keyword?
         :error-map map?))

(s/fdef exception
  :args ::exception-input
  :ret #(instance? clojure.lang.ExceptionInfo %)
  :fn (fn [{{:keys [status-code type]} :args ret :ret}]
        (= (select-keys (ex-data ret) [:exception/status-code :exception/type])
           {:exception/status-code status-code
            :exception/type        type})))
