(ns org.euandreh.http.components
  (:require [com.stuartsierra.component :as component]
            [io.pedestal.http :as http]
            [io.pedestal.interceptor :as i]
            [org.euandreh.http.components.crypto :as components.crypto]
            [org.euandreh.http.components.edn-database :as components.edn-database]
            [org.euandreh.http.components.edn-file :as components.edn-file]
            [tea-time.core :as tea-time]))

(defn new-edn-file [config-file-path & [config-ttl-path]]
  (components.edn-file/map->EdnFileImpl {:config-file-path config-file-path
                                         :config-ttl-path  config-ttl-path}))

(defn new-edn-database [config-path]
  (components.edn-database/map->EdnDatabaseImpl {:config-path config-path}))

(defn new-crypto []
  (components.crypto/map->JWSCryptoImpl {}))

(defn inject-components [webapp]
  (i/interceptor
   {:name  ::inject-components
    :enter #(assoc-in % [:request :components] webapp)}))

(defn- my-system-global-interceptors [service-map webapp]
  (update service-map
          ::http/interceptors
          (fn [interceptors-vector]
            (vec (cons (inject-components webapp) interceptors-vector)))))

(defn- test? [{:keys [env]}] (#{:test} env))
(defn- dev? [{:keys [env]}]  (#{:test :dev} env))

(defrecord Pedestal [routes service-map service config]
  component/Lifecycle
  (start [this]
    (if service
      this
      (let [config-map (merge {::http/port   (get config :port 1234)
                               ::http/routes routes
                               ::http/join?  false ;; do not block thread that starts web (server)
                               ::http/type   :jetty}
                              service-map)]
        (cond-> config-map
          true                     http/default-interceptors
          (dev? config-map)        http/dev-interceptors
          true                     (my-system-global-interceptors this)
          true                     http/create-server
          (not (test? config-map)) http/start
          true                     (#(assoc this :service %))))))
  (stop [this]
    (when (and service
               (not (test? service-map)))
      (http/stop service))
    (assoc this :service nil)))

(defn new-pedestal [routes]
  (map->Pedestal {:routes routes}))

(def default-pooling-sec 3)
(defrecord BackgroundJob [job-fn config-path deps config database]
  component/Lifecycle
  (start [this]
    (tea-time/start!)
    (let [wait-sec (get-in config config-path default-pooling-sec)
          every    (tea-time/every! wait-sec (partial job-fn (select-keys this deps)))]
      (assoc this :tea-time-every every)))
  (stop [this]
    (tea-time/cancel! (:tea-time-every this))
    (tea-time/stop!)
    (assoc this :tea-time-every nil)))

(defn new-background-job [job-fn config-path deps]
  (map->BackgroundJob {:job-fn      job-fn
                       :config-path config-path
                       :deps        deps}))
