(ns org.euandreh.misc.edn.urx
  "edn readers and custom printers for `lambdaisland.uri.URI` and `cemerick.url.URL`."
  (:require lambdaisland.uri
            cemerick.url
            org.euandreh.misc.spec.generators
            [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as s.gen])
  (:import lambdaisland.uri.URI
           cemerick.url.URL))

;; URI

(def string->uri
  "Builds and `lambdaisland.uri.URI` instance from a given string.

   Used for custom edn tags, placed inside `data_readers.clj` file."
  lambdaisland.uri/parse)

(defmethod print-method URI [uri ^java.io.Writer writer]
  (.write writer "#uri ")
  (print-method (str uri) writer))

(defmethod print-dup URI [uri ^java.io.Writer writer]
  (.write writer "#uri ")
  (print-method (str uri) writer))

;; URL

(defmethod print-method URL [url ^java.io.Writer writer]
  (.write writer "#url ")
  (print-method (str url) writer))

(defmethod print-dup URL [url ^java.io.Writer writer]
  (.write writer "#url ")
  (print-method (str url) writer))

(defn url?
  "Checks if the given input is a valid `cemerick.url.URL`."
  [x]
  (try
    (boolean (cemerick.url/url (str x)))
    (catch java.net.MalformedURLException e false)))

(defn string->url
  "Builds and `cemerick.url.URL` instance from a given string.

   Used for custom edn tags, placed inside `data_readers.clj` file."
  [url-string]
  (when (url? url-string)
    (cemerick.url/url url-string)))

;; Specs

(s/def ::url-string
  (s/with-gen (s/and string? url?)
    (fn [] org.euandreh.misc.spec.generators/url-string)))

(s/def ::url
  (s/with-gen url?
    (fn [] org.euandreh.misc.spec.generators/url)))

(s/fdef url?
  :args (s/alt :url-string ::url-string
               :url        ::url
               :any        any?)
  :ret boolean?
  :fn (fn [{[label _] :args ret :ret}]
        (cond
          (nil? ret)
          (= :any label)

          (#{:url :url-string} label)
          (= true ret)

          :otherwise
          true)))

(s/fdef string->url
  :args (s/alt :url-string ::url-string
               :url        ::url
               :string     string?)
  :ret (s/nilable url?)
  :fn (fn [{[label _] :args ret :ret}]
        (cond
          (nil? ret)
          (= :string label)

          (#{:url :url-string} label)
          (and (some? ret)
               (url? ret))

          :otherwise
          true)))
