(ns org.euandreh.misc.spec.generators
  "Custom generators for spec definitions and test.check tests."
  (:require [clojure.spec.gen.alpha :as s.gen]
            [clojure.spec.alpha :as s]
            cemerick.url))

;;; Misc

;; From the fact that (= ##NaN ##NaN) is false, it isn't usually very useful to have ##NaN inside a generated collectioon.
;; Further reference:
;; https://dev.clojure.org/jira/browse/CLJ-2054
;; https://stackoverflow.com/questions/9341653/float-nan-float-nan

(defn- replace-nan [x]
  (if (and (number? x)
           (Double/isNaN x))
    ::nan
    x))

(def any-without-nan
  "Same as `clojure.test.check.generators/any`, but without `##NaN` values.

   Since `(= ##NaN ##NaN)` is `false`, this generator is usually more useful for generating random values."
  (s.gen/fmap (partial clojure.walk/postwalk replace-nan) (s.gen/any)))

(s/fdef replace-nan
  :args (s/cat :x any?)
  :ret #(or (not (number? %))
            (false? (Double/isNaN %)))
  :fn (fn [{{x :x} :args ret :ret}]
        (if (and (number? x)
                 (Double/isNaN x))
          (= ret ::nan)
          true)))

;;; Instant

;; Derived from "Building test check Generators - Gary Fredericks", around minute 32
;; https://www.youtube.com/watch?v=F4VZPxLZUdA

(s/def ::year
  (s/with-gen int?
    #(s.gen/fmap (partial + 2017) (s.gen/int))))

(s/def ::month
  (s/with-gen (s/and int? #(<= 1 % 12))
    #(s.gen/large-integer* {:min 1 :max 12})))

(s/def ::day
  (s/with-gen (s/and int? #(<= 1 % 31))
    #(s.gen/large-integer* {:min 1 :max 31})))

(s/def ::hour
  (s/with-gen (s/and int? #(<= 0 % 23))
    #(s.gen/large-integer* {:min 0 :max 23})))

(s/def ::minute
  (s/with-gen (s/and int? #(<= 0 % 59))
    #(s.gen/large-integer* {:min 0 :max 59})))

(s/def ::second
  (s/with-gen (s/and int? #(<= 0 % 59))
    #(s.gen/large-integer* {:min 0 :max 59})))

(s/def ::millis
  (s/with-gen (s/and int? #(<= 0 % 999))
    #(s.gen/large-integer* {:min 0 :max 999})))

(s/def ::instant-map
  (s/keys
   :req [::year ::month ::day ::hour ::minute ::second ::millis]))

(defn- construct-instant
  "Not to be used for date parsing, just a helper function for generating useful instants."
  [instant-map]
  (letfn [(try-construct-instant
            [{::keys [year month day hour minute second millis] :as instant-map-inner}
             try-n]
            (when (> try-n 4)
              (throw (ex-info "Max tries (4) to build an Instant, invalid :day entry."
                              {:try-n        try-n
                               :original-day (+ (dec try-n) day)
                               :instant-map  instant-map-inner})))
            (let [format-str "%04d-%02d-%02dT%02d:%02d:%02d.%03dZ"]
              (try
                (java.time.Instant/parse
                 (format format-str
                         year month day hour minute second millis))
                (catch java.time.format.DateTimeParseException e
                  (try-construct-instant (update instant-map-inner ::day dec) (inc try-n))))))]
    (try-construct-instant instant-map 1)))

(s/fdef construct-instant
  :args (s/cat :instant-map ::instant-map)
  :ret #(instance? java.time.Instant %))

(def instant
  "Custom `java.time.Instant` generator, compatible with `clojure.test.check`."
  (s.gen/fmap construct-instant (s/gen ::instant-map)))

;;; URI

;; Derived from "A spec for URLs in Clojure"
;; http://conan.is/blogging/a-spec-for-urls-in-clojure.html

(def alphanumeric-regex
  "Regex for 0 or more alphanumeric chars."
  #"^[a-zA-Z0-9]*$")

(s/def ::alphanumeric-string
  (s/with-gen (s/and string? #(re-matches alphanumeric-regex %))
    #(s.gen/string-alphanumeric)))

(s/def ::non-empty-alphanumeric-string (s/and ::alphanumeric-string #(not= "" %)))
(s/def ::protocol #{"http" "https"})
(s/def ::port (s/and int? #(<= 1 % 65535)))

(s/def ::username ::alphanumeric-string)
(s/def ::password ::alphanumeric-string)
(s/def ::anchor   ::alphanumeric-string)
(s/def ::host     ::alphanumeric-string)
(s/def ::anchor   ::alphanumeric-string)

(def path-regex
  "URL path: concatenation of alphanumeric chars plus '/' chars."
  #"^[a-zA-Z0-9/]*$")

(s/def ::path-parts (s/coll-of ::non-empty-alphanumeric-string))

(defn- build-path
  [strings-vector]
  (->> strings-vector
       (interleave (repeat "/"))
       (apply str)))

(s/fdef build-path
  :args (s/cat :path-parts ::path-parts)
  :ret #(re-matches path-regex %)
  :fn #(string? (:ret %)))

(s/def ::path
  (s/with-gen (s/and string? #(re-matches path-regex %))
    #(s.gen/fmap build-path
                 (s/gen ::path-parts))))

(s/def ::query
  (s/map-of ::non-empty-alphanumeric-string ::non-empty-alphanumeric-string))

(s/def ::url-map
  (s/keys
   :req [::protocol ::host ::path]
   :opt [::username ::password ::port ::query ::anchor]))

(defn- construct-url
  "Build a `cemerick.url/URL from a ::url-map input."
  [{::keys [protocol username password host port path query anchor]}]
  (cemerick.url/->URL protocol username password host port path query anchor))

(def url
  "Custom `cemerick.url/URL` generator, compatible with `clojure.test.check`."
  (s.gen/fmap construct-url (s/gen ::url-map)))

(def url-string
  "Custom generator for the string content of a `cemerick.url/URL`, compatible with `clojure.test.check`."
  (s.gen/fmap (comp str construct-url) (s/gen ::url-map)))
