(ns org.euandreh.misc.edn.email
  "edn readers and custom printers for Email."
  (:require [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as s.gen]
            [miner.strgen :as strgen]
            [org.euandreh.misc.spec.utils :as misc.spec.utils]))

;; Taken from:
;; https://clojure.org/guides/spec#_entity_maps
(def email-regex
  "Email regex used for parsing strings. Shouldn't be ultimately relied upon, just a pretty add-on."
  #"^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,63}$")

(defrecord Email [email-string]
  Object
  (toString [_] email-string))

(defn string->email
  "Build an `Email` instance from a `email-string` input.

   Used for custom edn tags, placed inside `data_readers.clj` file."
  [email-string]
  (misc.spec.utils/conform! ::email-string email-string)
  (->Email email-string))

(defmethod print-method Email [email ^java.io.Writer writer]
  (.write writer "#email ")
  (print-method (str email) writer))

(defmethod print-dup Email [email ^java.io.Writer writer]
  (.write writer "#email ")
  (print-method (str email) writer))

;; Specs

(s/def ::email-string
  (s/with-gen (s/and string? #(re-matches email-regex %))
    #(strgen/string-generator email-regex)))

(s/def ::email-instance
  (s/with-gen #(instance? Email %)
    #(s.gen/fmap string->email
                 (s/gen ::email-string))))

(s/fdef string->email
  :args (s/cat :email-string ::email-string)
  :ret  ::email-instance)
