(ns ezand.flume-engine.engines.root-engine
  "
  Clojure port of the original Flume Node JS `RootEngine`:
  https://github.com/chrisjpatty/flume/blob/master/src/RootEngine.js
  "
  (:require [clojure.tools.logging :as log]
            [ezand.flume-engine.utils :as util]
            [slingshot.slingshot :refer [try+ throw+]]))

(def ^:private ^:const default-max-loops 1000)

;;;;;;;;;;;;;;;;;;;;
;; Error Handling ;;
;;;;;;;;;;;;;;;;;;;;
(def ^:private ^:const max-loops-exceeded-error
  {:type :root-engine/max-loops-exceeded
   :message "Max loop count exceeded."})

(def ^:private ^:const root-node-error
  {:type :root-engine/root-node-error
   :message "A root node was not found. The Root Engine requires that exactly one node be marked as the root node."})

(defn- ensure-root-node
  [root-node]
  (if-not root-node
    (throw+ root-node-error)
    root-node))

(defn- ensure-loop-count
  [max-loops loops]
  (if (and (> max-loops 0)
           (> @loops max-loops))
    (throw+ max-loops-exceeded-error)
    (swap! loops inc)))

;;;;;;;;;;;;;;;;;
;; Root Engine ;;
;;;;;;;;;;;;;;;;;
(defn- get-root-node
  [nodes]
  (ensure-root-node
    (util/find-first #(-> % :root true?) (vals nodes))))

(defn- reduce-root-inputs
  [inputs callback]
  (reduce-kv (fn [acc input-name connection]
               (let [input (callback input-name connection)]
                 (assoc acc (:name input) (:value input))))
             {}
             inputs))

(defn- get-value-of-connection
  [config max-loops loops
   resolve-input-controls-fn
   fire-node-fn
   {:keys [node-id port-name] :as connection}
   nodes context]
  (ensure-loop-count max-loops loops)
  (letfn [(resolve-input-values [{:keys [connections input-data] :as node}
                                 {node-type-inputs :inputs :as node-type}
                                 nodes context]
            (let [connections-inputs (:inputs connections)]
              (reduce (fn [acc {input-type :type
                                input-name :name :as input}]
                        (let [input-connections (get connections-inputs input-name)]
                          (if (and input-connections (not-empty input-connections))
                            (assoc acc input-name (get-value-of-connection config max-loops loops resolve-input-controls-fn fire-node-fn (first input-connections) nodes context))
                            (assoc acc input-name (resolve-input-controls-fn input-type (get input-data input-name {}) context)))))
                      {}
                      node-type-inputs)))]
    (let [output-node (get nodes node-id)
          output-node-type (get-in config [:node-types (:type output-node)])
          input-values (resolve-input-values output-node output-node-type nodes context)
          output-result (fire-node-fn output-node input-values output-node-type connection)]
      (get output-result port-name))))

(defn resolve-root-node
  [config nodes
   resolve-input-controls-fn
   fire-node-fn
   {root-node-id :root-node-id
    context :context
    max-loops :max-loops
    only-resolve-connected? :only-resolve-connected
    :or {max-loops default-max-loops} :as options}]
  (let [loops (atom 0)
        {node-type :type :as root-node} (-> (or (get nodes root-node-id)
                                                (get-root-node nodes))
                                            (ensure-root-node))
        inputs (get-in config [:node-types node-type :inputs])
        control-values (reduce (fn [acc {input-name :name
                                         input-type :type :as input}]
                                 (let [input-data (get-in root-node [:input-data input-name] {})]
                                   (assoc acc (keyword input-name)
                                              (resolve-input-controls-fn input-type input-data context))))
                               {}
                               inputs)
        input-values (reduce-root-inputs
                       (get-in root-node [:connections :inputs])
                       (fn [input-name connection]
                         (reset! loops 0)
                         (let [input-value (try+
                                             (get-value-of-connection config max-loops loops resolve-input-controls-fn fire-node-fn (first connection) nodes context)
                                             (catch [:type :root-engine/max-loops-exceeded] {:keys [message]}
                                               (log/error message))
                                             (catch any? e
                                               (log/error e "Failed to resolve input values.")))]
                           {:name input-name
                            :value input-value})))]
    (if only-resolve-connected?
      input-values
      (merge control-values input-values))))
