(ns hypercrud-service.rest.routes
  (:require [hypercrud-service.collection-json :as cj]
            [hypercrud-service.rest.crud :as crud]
            [hypercrud-service.rest.index :as index]
            [hypercrud-service.response :as response]
            [hypercrud-service.datomic-util :refer [latest-tx]]
            [hypercrud-service.rest.datomic-crud]))



;; These should operate on string types, because different database backends might parse them differently.
;; For example datomic id is a long, but other objects might be a uuid.

(defn mk-coll-get [coll-route query-fn typetag]
  (fn [{{:keys [tx]} :query-params}]
    (let [tx (Long/parseLong tx)     ;tx is a hypercrud parameter so it's always a Long
          resultset (query-fn tx)
          typeinfo (cj/typeinfo typetag tx)
          links []]
      (response/->ReadCollectionResponse coll-route links resultset typeinfo typetag tx))))


(defn mk-coll-post [typetag]
  (fn [{:keys [:hypercrud-service.pedestal-util/body-params]}]
    ;; Should we also check tx (to detect concurrent modifications of collection)?
    ;; I don't think that makes sense on item creation
    (-> (crud/entity-create typetag body-params)
        (response/->CreateItemResponse))))


(defn make-item-get [typetag {{:keys [tx]} :query-params {:keys [id]} :path-params}]
  (let [tx (Long/parseLong tx)
        record (crud/entity-read typetag id tx)  ;id stays a string- crud plugin will parse, e.g. it could be a uuid
        typeinfo (cj/typeinfo typetag tx record)
        links []]
    (response/->ReadItemResponse links record typeinfo typetag tx)))


(defn mk-item-put [typetag]
  (fn [{{:keys [tx]} :query-params
        {:keys [id]} :path-params
        :keys [:hypercrud-service.pedestal-util/body-params]}]
    (let [new-tx (crud/entity-update typetag id (Long/parseLong tx) body-params)]
      (response/->UpdateItemResponse new-tx))))


(defn mk-index-get [route-name endpoints]
  (fn [{{:keys [tx]} :query-params}]
    (let [tx (if tx (Long/parseLong tx) (latest-tx))
          links (index/link-index endpoints tx)
          records []
          typeinfo nil
          typetag nil]
      (response/->ReadCollectionResponse route-name links records typeinfo typetag tx))))


(defn mk-inner-routes [root query-fn typetag]
  ;; see: https://groups.google.com/forum/#!topic/pedestal-users/UXezozhUFM4
  (let [coll-route (keyword root)
        item-route (keyword (str root "-item"))]
    `[~(str "/" root) {:get [~coll-route (mk-coll-get ~coll-route '~query-fn ~typetag)]
                       :post [~(keyword root "post") (mk-coll-post ~typetag)]}
      ["/:id" {:get [~item-route (partial make-item-get ~typetag)]
               :put [~(keyword (str root "-item") "put") (mk-item-put ~typetag)]}]]))