(ns hypercrud.service.core
  (:require [datomic.api :as d]
            [datomico.db :as db]
            [hypercrud.service.datomic-util :as datomic-util :refer [with-db-as-of]]
            [hypercrud.service.pedestal-util :as pedestal-util]
            [hypercrud.service.representation]
            [hypercrud.service.response :as response]
            [hypercrud.service.root-renderer :as root-renderer]
            [io.pedestal.http.body-params :as body-params]))


(defn query [model
             {{:keys [tx]} :query-params
              {:keys [queryname]} :path-params}]
  (let [tx (Long/parseLong tx)
        q (get-in model [:queries (keyword queryname)])
        resultset (datomic-util/datomic-simple-q q tx)]
    (response/->QueryResponse resultset)))


(defn entity [model
              {{:keys [tx]} :query-params
               {:keys [eid]} :path-params}]
  (let [tx (Long/parseLong tx)
        eid (Long/parseLong eid)
        entity (with-db-as-of tx (d/entity db/*db* eid))
        type ((:reflect-entity-type model) entity)]
    (response/->EntityResponse entity (get-in model [:forms type]) tx)))


(defn transact! [{:keys [::pedestal-util/body-params]}]
  (let [effect (d/transact db/*connection* body-params)
        tx (-> @effect :db-after d/basis-t d/t->tx)]
    (response/->TransactResponse tx)))


(defn index [request]
  (response/->IndexResponse (datomic-util/latest-tx)))


(defn route [root model]
  `[~root {:get [:index index]}
    ^:interceptors [~(body-params/body-params)
                    pedestal-util/auto-content-type
                    pedestal-util/combine-body-params
                    pedestal-util/wrap-ring-response
                    root-renderer/wrap-hypercrud]

    ["/query/:queryname" {:get [:query (partial query ~model)]}]
    ["/entity/:eid" {:get [:entity (partial entity ~model)]}]
    ["/transact" {:post [:transact transact!]}]])
