(ns hypercrud.service.core
  (:require [datomic.api :as d]
            [datomico.db :as db]
            [hypercrud.service.datomic-util :as datomic-util :refer [with-db-as-of]]
            [hypercrud.service.pedestal-util :as pedestal-util]
            [hypercrud.service.root-renderer :as root-renderer]
            [io.pedestal.http.body-params :as body-params]))


;; Nice to render out vectors for the client since they are already associative and any
;; benefits of list e.g. lazy are lost due to serialization on the wire


(defn query [model
             {{:keys [tx]} :query-params
              {:keys [queryname]} :path-params}]
  (let [tx (Long/parseLong tx)
        q (get-in model [:queries (keyword queryname)])
        resultset (datomic-util/datomic-simple-q q tx)]
    (with-db-as-of tx
      (mapv (comp :db/id (partial d/entity db/*db*)) resultset))))


(defn represent-ref-val [v]
  ;; don't d/entity an entity
  (:db/id (if (keyword? v) (d/entity db/*db* v) v)))


(defn entity [model
              {{:keys [tx]} :query-params
               {:keys [eid]} :path-params}]
  (let [tx (Long/parseLong tx)
        eid (Long/parseLong eid)
        entity (with-db-as-of tx (d/entity db/*db* eid))
        type ((:reflect-entity-type model) entity)
        form (get-in model [:forms type])]
    (with-db-as-of tx
      (->> form
           (map (fn [{:keys [name datatype set]}]
                  (let [v (get entity name)
                        v (cond

                            (and (= datatype :ref) (not set) (not (nil? v)))
                            (represent-ref-val v)

                            (and (= datatype :ref) set (not (nil? v)))
                            (into #{} (map represent-ref-val v))

                            :not-a-ref
                            v)]
                    [name v])))
           (into (select-keys entity [:meta/type]))))))


(defn transact! [{:keys [::pedestal-util/body-params]}]
  (let [effect (d/transact db/*connection* body-params)
        tx (-> @effect :db-after d/basis-t d/t->tx)]
    {:tx tx}))


(defn index [request]
  {:tx (datomic-util/latest-tx)})


(defn route [root model]
  `[~root {:get [:index index]}
    ^:interceptors [~(body-params/body-params)
                    pedestal-util/auto-content-type
                    pedestal-util/combine-body-params
                    pedestal-util/wrap-ring-response
                    root-renderer/wrap-hypercrud]

    ["/query/:queryname" {:get [:query (partial query ~model)]}]
    ["/entity/:eid" {:get [:entity (partial entity ~model)]}]
    ["/transact" {:post [:transact transact!]}]])
