(ns hypercrud.service.core
  (:require [clojure.core.match :refer [match]]
            [datomic.api :as d]
            [datomico.db :as db]
            [hypercrud.service.datomic-util :as datomic-util :refer [with-db-as-of]]
            [hypercrud.service.pedestal-util :as pedestal-util]
            [io.pedestal.http.body-params :as body-params]
            [ring.util.response :as ring-resp]))


;; Nice to render out vectors for the client since they are already associative and any
;; benefits of list e.g. lazy are lost due to serialization on the wire


(defn wrap-hypercrud [m]
  {:hypercrud m})


(defn query [{{:keys [tx]} :query-params
              q :body-params}]
  (let [tx (Long/parseLong tx)]
    (with-db-as-of tx
      (->> (datomic-util/datomic-simple-q q tx)
           (mapv (comp :db/id (partial d/entity db/*db*)))
           wrap-hypercrud
           ring-resp/response))))


(defn represent-ref-val [v]
  ;; don't d/entity an entity
  (:db/id (if (keyword? v) (d/entity db/*db* v) v)))


(defn entity [schema
              {{:keys [tx]} :query-params
               {:keys [eid]} :path-params}]
  (let [tx (Long/parseLong tx)
        eid (Long/parseLong eid)
        entity (with-db-as-of tx (d/entity db/*db* eid))]
    (with-db-as-of tx
      (->> entity
           (map (fn [[attr val]]
                  [attr (match ((juxt :db/valueType :db/cardinality) (attr schema))
                               [:db.type/ref :db.cardinality/one] (represent-ref-val val)
                               [:db.type/ref :db.cardinality/many] (into #{} (map represent-ref-val val))
                               [_ _] val)]))
           (into {})
           wrap-hypercrud
           ring-resp/response))))


(defn transact! [{:keys [:body-params]}]
  (let [effect (d/transact db/*connection* body-params)
        tx (-> @effect :db-after d/basis-t d/t->tx)]
    (ring-resp/response (wrap-hypercrud {:tx tx}))))


(defn index [request]
  (ring-resp/response (wrap-hypercrud {:tx (datomic-util/latest-tx)})))


(defn route [root schema]
  `[~root {:get [:index index]}
    ^:interceptors [~(body-params/body-params)
                    pedestal-util/combine-body-params
                    pedestal-util/auto-content-type]

    ["/query" {:post [:query query]}]
    ["/entity/:eid" {:get [:entity (partial entity ~schema)]}]
    ["/transact" {:post [:transact transact!]}]])
