(ns hypercrud.service.core
  (:require [clojure.core.match :refer [match]]
            [datomic.api :as d]
            [hypercrud.service.pedestal-util :as pedestal-util]
            [io.pedestal.http.body-params :as body-params]
            [ring.util.response :as ring-resp]))


;; Nice to render out vectors for the client since they are already associative and any
;; benefits of list e.g. lazy are lost due to serialization on the wire


(defn wrap-hypercrud [m]
  {:hypercrud m})


(defn query [datomic-conn
             {{:keys [tx]} :query-params
              [query params] :body-params}]
  (let [tx (Long/parseLong tx)
        db (d/as-of (d/db datomic-conn) tx)]
    (->> (apply d/q query db params)
         (mapv (comp :db/id (partial d/entity db)))         ;resolve idents
         wrap-hypercrud
         ring-resp/response)))


(defn represent-ref-val [db v]
  ;; don't d/entity an entity
  (:db/id (if (keyword? v) (d/entity db v) v)))


(defn entity [datomic-conn
              schema
              {{:keys [tx]} :query-params
               {:keys [eid]} :path-params}]
  (let [tx (Long/parseLong tx)
        eid (Long/parseLong eid)
        db (d/as-of (d/db datomic-conn) tx)]
    (->> (d/entity db eid)
         (map (fn [[attr val]]
                [attr (match ((juxt :db/valueType :db/cardinality) (attr schema))
                             [:db.type/ref :db.cardinality/one] (represent-ref-val db val)
                             [:db.type/ref :db.cardinality/many] (into #{} (map #(represent-ref-val db %) val))
                             [_ _] val)]))
         (into {:db/id eid})
         wrap-hypercrud
         ring-resp/response)))


(defn transact! [datomic-conn
                 {:keys [:body-params]}]
  (let [effect (d/transact datomic-conn body-params)
        tx (-> @effect :db-after d/basis-t d/t->tx)]
    (ring-resp/response (wrap-hypercrud {:tx tx}))))


(defn index [datomic-conn request]
  (let [tx (-> (d/db datomic-conn) d/basis-t d/t->tx)]
    (ring-resp/response (wrap-hypercrud {:tx tx}))))


(defn latest [datomic-conn request]
  (let [tx (-> (d/db datomic-conn) d/basis-t d/t->tx)]
     (ring-resp/response (str tx))))


(defn route [datomic-conn root schema]
  `[~root {:get [:index (partial index ~datomic-conn)]}
    ^:interceptors [~(body-params/body-params)
                    pedestal-util/combine-body-params
                    pedestal-util/auto-content-type]

    ["/query" {:post [:query (partial query ~datomic-conn)]}]
    ["/entity/:eid" {:get [:entity (partial entity ~datomic-conn ~schema)]}]
    ["/transact" {:post [:transact (partial transact! ~datomic-conn)]}]
    ["/latest" {:get [:latest (partial latest ~datomic-conn)]}]])
